;; *************************************************************************************** ;;
;; Jabberwocky  : Programming envioronment for LISP                                        ;;
;; Copyright (C) 1998 Marc Mertens                                                         ;;
;;                                                                                         ;;
;;     This program is free software; you can redistribute it and/or modify                ;;
;;    it under the terms of the GNU General Public License as published by                 ;;
;;    the Free Software Foundation; either version 2 of the License, or                    ;;
;;    (at your option) any later version.                                                  ;;
;;                                                                                        ;;
;;    This program is distributed in the hope that it will be useful,                      ;;
;;    but WITHOUT ANY WARRANTY; without even the implied warranty of                       ;;
;;    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                        ;;
;;    GNU General Public License for more details.                                         ;;
;;                                                                                         ;;
;;    You should have received a copy of the GNU General Public License                    ;;
;;    along with this program; if not, write to the Free Software                          ;;
;;    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA            ;;
;;                                                                                         ;;
;; Contact me on mmertens@vt4.net                                                          ;;
;; *************************************************************************************** ;;
;; 
;; The code is divided  in 5 sections 
;;
;; Section 1 : Interface code to the c program which represent the GUI
;; Section 2 : Code to transform a source in a source with debugged code added 
;; Section 3 : Code to process a syntax file , to generate the transforming functions needed
;;             in section 2
;; Section   : Code to generate java code for the syntax analysis of the java editor
;; Section 5 : Code used during the actual debugging process
;; *************************************************************************************** ;;

;; *************************************************************************************** ;;
;; Package stuff                                                                           ;;
;; *************************************************************************************** ;;
#-(or :SBCL GCL)
(when (not (find-package "DEBUGGER")) (make-package "DEBUGGER"))
#+(or :SBCL GCL)
(when (not (find-package "DEBUGGER")) (make-package "DEBUGGER" :use '("COMMON-LISP" "COMMON-LISP-USER")))
        
(in-package "DEBUGGER")

;; *************************************************************************************** ;;
;; Optimize commands for the compiler                                                      ;;
;; *************************************************************************************** ;;

#+:CMU
(declaim (optimize (speed 3) (safety 0)) (extensions:optimize-interface (speed 3)))
#+gcl
(declaim (optimize (speed 3) (safety 0)))
#+CLISP
(declaim (optimize (speed 3) (safety 0)))


;; ##################################################
;; ##################################################
;; ##################################################
;; ###### SECTION 1 #################################
;; ##################################################
;; ##################################################
;; ##################################################

;; *************************************************************************************
;; Lisp implementation dependent definitions
;; *************************************************************************************

#+:CMU
(load "cmucl/cmu.x86f")
#+:SBCL
(load "sbcl/sbcl.fasl")
#+CLISP
(load "clisp/clisp.fas")
#+:acl-socket
(load "acl/acl.fasl")
#+GCL
(load "gcl/gcl.o")

;; *************************************************************************************
;; Link to the java interface
;; *************************************************************************************

#+:CMU
(load "cmucl/java.x86f")
#+:SBCL
(load "sbcl/java.fasl")
#+CLISP
(load "clisp/java.fas")
#+:acl-socket
(load "acl/java.fasl")
#+GCL
(load "gcl/java.o")

;;; ***************************************************************************
;;; Class to hold the debugger state
;;; breakpoints  hashtable mapping (begin end) to breakpoint condition or T
;;; next-breakpoint, (begin end) next breakpoint setting 
;;; top-list (begin end) the toplist  of the previous call
;;; call-count, hashtable containing counts for the execution of forms
;;; ***************************************************************************

(defclass debugger-state ()
  ((breakpoints :initform (make-hash-table :test #'equal) :reader breakpoints)
   (next-breakpoint :initform nil :accessor next-breakpoint)
   (top-list :initform nil :accessor top-list)
   (call-count :initform nil :accessor call-count)))

;; ****************************************************************************
;; Constants used in this code
;; ****************************************************************************
(when (not (constantp '++common-lisp-user++))
  #+CLISP
  (defconstant ++common-lisp-user++ "COMMON-LISP-USER")
  #+:CMU
  (defconstant ++common-lisp-user++ "COMMON-LISP-USER")
  #+:SBCL
  (defconstant ++common-lisp-user++ "COMMON-LISP-USER")
  #+:acl-socket
  (defconstant ++common-lisp-user++ "common-lisp-user")
  #+GCL
  (defconstant ++common-lisp-user++ "COMMON-LISP-USER"))


;; ************************************************************************************** ;;
;; Global control variables                                                               ;;
;; ************************************************************************************** ;;

(defparameter **alternate-time** -1) ;; The alternate time
(defparameter **now** 0)  ;; The now in timetraveling
(defparameter **end-of-time** 0) ;; The end of time in timetraveling
(defparameter **debug-macro** nil) ;; Allow / disallow generation of debugging code for macros
(defparameter **connection** nil) ;; Connection to the java interface
(defparameter **interaction** nil) ;; Interaction object in java
(defparameter **breakpoints** (make-hash-table :test #'equal)) ;; Contains all the possible breakpoints
(defparameter **load** nil) ;; Indicates that we are in a load situation,used to avoid activating the debugger during load
(defparameter **option** (gensym)) ;; Represents a #+ or #- option
(defparameter **check-error** t) ; Indication if during evaluation of a call, checking of errors must be done
(defparameter **debug-after** t) ; Indicate that we want to get the result after a call
(defparameter **step-into** nil) ; Indicate that we do a stepint
(defparameter **editors** (make-hash-table :test #'equal)) ; map of editor to a debugger-status for this editor
(defparameter **profile** nil) ; Indicate if call counting must be do
(defparameter **watches** nil) ; List of (node watchexpression) (watchpoints to display)
(defparameter **watch-node-result** nil) ; Indicates the node used for reporting the resul of evaluating a form

;; ****************************************************************
;; Global vars used in debugging section
;; ****************************************************************

(defparameter **current-env** nil)                              ;; Save of environment in debug-point
(defparameter **current-begin** 0)                              ;; Save of begin point in source of debug-point
(defparameter **current-end** 0)                                ;; Save of end point in source of debug-point
(defparameter **current-source** "")                            ;; Save of source in debug-point
(defparameter **end-debug-eventloop** nil)                      ;; Indicator to stay/leave debugger eventloop
(defparameter **time** nil)                                     ;; Used in timetraveling to hold time
(defparameter **display-result-call** nil)                      ;; Indicator used to show result of call in debugpoint
(defparameter **watchpoints** nil)                              ;; List of watchpoints
(defparameter **values** nil)                                   ;; New values to be returned instead of the stated ones
(defparameter **after-separate** nil)                           ;; Indication if after result is to be displayed in separate window or in the resultpane
(defparameter **run-error** nil)                                ;; Set to true if there is a runtime error during execution debugged code

;; *********************************************************
;; System independend function to interact with interface ;;
;; *********************************************************
;;                                                        ;;
;; Here comes the real code to the debugger               ;; 
;;                                                        ;;
;; *********************************************************

;; *********************************************************
;; Used by java to set the **interaction** interface
;; *********************************************************

(defun set-interaction (interaction)
  (setf **interaction** interaction))

(defun enter-debug (port1 port2)
  (prepare-time 100) ;; prepare timetravelling to hold a max of 100 entries
  (setf **watchpoints** nil)
  ;; Load the conversion functions of the debugger
  #+:CMU
  (load "cmucl/debugcode.x86f") ;; Load the syntax analyzer (cmucl)
  #+:SBCL
  (load "sbcl/debugcode.fasl")
  #+GCL
  (load "gcl/debugcode.o")
  #+CLISP
  (load "clisp/debugcode.fas") ;; Load the syntax analyzer (clisp)
  #+:acl-socket
  (load "acl/debugcode.fasl") ;; Load the syntax analyzer (acl)
  #+:CMU
  (load "cmucl/jabberwocky.x86f") ;; Load the proxy java functions (cmucl)
  #+:SBCL
  (load "sbcl/jabberwocky.fasl")
  #+GCL
  (load "gcl/jabberwocky.o")
  #+CLISP
  (load "clisp/jabberwocky.fas") ;; Load the proxy java functions (clisp)
  #+:acl-socket
  (load "acl/jabberwocky.fasl") ;; Load the proxy java function (acl)
  ;; Connect to the graphical interface
  (setf **connection** (make-java-connection port1 port2))
  ;; Define the process incoming fucntion
  (setf (symbol-function 'process-incoming) (java-connection-handler **connection**))
  ;; Call process-incoming for the first time
  (process-incoming)
  #+GCL
  (start-listen-read-eval **connection**)
  )

;; *********************************************************
;; Send a message in a message box                        ;;
;; *********************************************************

(defun message (message &key ((:warn warn) nil))
  (if **connection**
      (if warn
          (call-java-method **connection**
                            "IDE.Messaging.MessageHandler"
                            "warn"
                            message)
        (call-java-method **connection**
                          "IDE.Messaging.MessageHandler"
                          "say"
                          message))
    (format t "**** Error : ~A ****" message)))

;; *********************************************************
;; Highlight line of error in the source                  ;;
;; *********************************************************

(defun highlight-error (begin)
  (call-java-method **connection**
             (deserialize-java-object (db-c-source))
                 "setCursor"
             begin))

;; *********************************************************
;; Set possible breakpoints in interface                  ;;
;; *********************************************************


(defun set-possible-breakpoint (editor begin end)
  (let ((breakpoints-editor (gethash editor **breakpoints**)))
    (when (null breakpoints-editor)
      (setf breakpoints-editor (make-hash-table :test #'equal))
      (setf (gethash editor **breakpoints**) breakpoints-editor))
    (setf (gethash (list begin  end) breakpoints-editor) t)))

;; *****************************************************************
;; Test if (source begin end) forms a possible breakpoint
;; *****************************************************************

(defun is-breakpoint (editor begin end)
  (declare (fixnum begin end))
  (let ((breakpoints-editor (gethash (serialize-java-object editor) **breakpoints**)))
    (if breakpoints-editor
        (if (gethash (list begin end) breakpoints-editor)
            t
          nil))))

;;; *****************************************************************
;;; To avoid a call to the interface for each call to debug-point
;;; we test first if we must call the interface or not
;;; for this raison we must set breakpoints also in the lisp process
;;; *****************************************************************

;;; ***************************************************************************
;;; Get the debugger-state, creating it if needed, a state is thus always 
;;; returned
;;; ***************************************************************************

(defun get-debugger-state (editor)
  (let ((debugger-state (gethash editor **editors**)))
    (declare (fixnum editor))
    (cond (debugger-state debugger-state)
          (t
           (setf debugger-state (make-instance 'debugger-state))
           (setf (gethash editor **editors**) debugger-state)
           debugger-state))))


;;; **************************************************************************
;;; Toggle a active breakpoint (called from the interface)
;;; Returns 1  if a breakpoint is set,
;;;         -1 if a breakpoint was unset
;;;         0  if it was a invalid breakpoint
;;; **************************************************************************

(defun toggle-breakpoint (editor begin end &optional (condition t))
  (let ((debugger-state (get-debugger-state (serialize-java-object editor))))
    (cond ((not (is-breakpoint editor begin end))
           0)
          (t
           (let ((con (gethash (list begin end) 
                               (breakpoints debugger-state))))
             (cond (con
                    (remhash (list begin end) (breakpoints debugger-state))
                    -1) ; Breakpoint has been removed
                   (t
                    (setf (gethash (list begin end) 
                                   (breakpoints debugger-state))
                          condition)
                    1))))))) ; Breakpoint has been set

;;; **************************************************************************
;;; Register a watch expression so it can be shown when we give control to 
;;; the interface
;;; **************************************************************************

(defun register-watch-expression (node watch-expression)
  (pushnew (list (serialize-java-object node) watch-expression) **watches**)
  (refresh-watches))

;;; **************************************************************************
;;; Remove a watch expression 
;;; **************************************************************************

(defun unregister-watch-expression (node)
  (setf **watches**
        (remove-if #'(lambda (el) (eql (first el) (serialize-java-object node)))
                   **watches**))
  (refresh-watches))

;;; **************************************************************************
;;; Replace the watch expression for a node by another
;;; **************************************************************************

(defun change-watch-expression (node watch-expression)
  (setf **watches** (remove-if #'(lambda (el) (eq (first el)
                                                  (serialize-java-object node)))
                               **watches**))
  (pushnew (list (serialize-java-object node) watch-expression) **watches**)
  (refresh-watches))
  
;;; **************************************************************************
;;; Start profiling (called from the interface) 
;;; **************************************************************************

(defun start-profiling (editor)
  (setf (call-count (get-debugger-state (serialize-java-object editor)))
        (make-hash-table :test #'equal))
  (setf **profile** t))

;;; **************************************************************************
;;; Stop profiling (called from the interface)
;;; **************************************************************************

(defun stop-profiling (editor)
  (setf (call-count (get-debugger-state (serialize-java-object editor)))
        nil)
  ;; Check if call counting must still be done for a editor
  (setf **profile** nil)
  (maphash #'(lambda (editor debugger-state)
                     (when (call-count debugger-state)
                       (setf **profile** t)
                       (return-from stop-profiling))) 
           **editors**))


;;; **************************************************************************
;;; Used by debug-point to count the number of times that a form is called
;;; Update the count only when call counting for the specified editor is 
;;; enable
;;; **************************************************************************

(defun register-call (editor begin end)
  (let ((debugger-state (gethash editor **editors**))
        (call-count nil))
    (declare (integer call-count))
    (when debugger-state
      (setf call-count (call-count debugger-state))
      (when call-count
        (setf (gethash (list begin end) (call-count debugger-state))
              (+ 1 (or (gethash (list begin end) (call-count debugger-state)) 
                       0)))))))

;;; **************************************************************************
;;; Called by the interface to get the call count for a given editor
;;; **************************************************************************

(defun get-call-count (editor)
  (let ((result nil)
        (debugger-state (gethash (serialize-java-object editor) **editors**))
        (call-count nil))
    (when debugger-state
      (setf call-count (call-count debugger-state))
      (when call-count
        (maphash #'(lambda (key value) 
                           (push (list (first key) (second key) value) result))
                 call-count)))
    result))

;;; **************************************************************************
;;; Used by debug-point to check if we must stop after a call of a form
;;; **************************************************************************

(defun must-stop-after-call (editor begin end)
  (let ((debugger-state (gethash editor **editors**))
        (condition nil)
        (top-list nil))
    (cond (debugger-state
           ;; Check if we really must stop
           (setf condition (gethash (list begin end) 
                                    (breakpoints debugger-state)))
           (setf top-list (top-list debugger-state))
           (or
            ;; We do a stepinto
            **step-into**
            ;; If we have a (conditional) breakpoint we stop
            (and condition (or (eq condition t) (expression-evaluate-to-true condition)))
            ;; Check if we have reached a next breakpoint
            (equal (next-breakpoint debugger-state) (list begin end))
            ;; Check for a stepover condition
            (and top-list 
                 (<= (first top-list) begin) 
                 (<= end (second top-list)))))
          (t nil))))

;;; ***************************************************************************
;;; Called by the interface for the step-into action
;;; ***************************************************************************

(defun step-into-action (editor)
  (let ((debugger-state (get-debugger-state (serialize-java-object editor))))
    (setf **step-into** t)
    (setf (top-list debugger-state) nil)
    (setf (next-breakpoint debugger-state) nil)
    (end-debug-eventloop)))

;;; ****************************************************************************
;;; Called by the interface for the step-over action
;;; ****************************************************************************

(defun step-over-action (editor &optional (begin-top-list nil) (end-top-list nil))
  (let ((debugger-state (get-debugger-state (serialize-java-object editor))))
    (setf **step-into** nil)
    (setf (next-breakpoint debugger-state) nil)
    (setf (top-list debugger-state)
          (if begin-top-list
              (list begin-top-list end-top-list)
            nil))
    (end-debug-eventloop)))

;;; ***************************************************************************
;;; Called by the interface for the stop-after action
;;; ***************************************************************************

(defun stop-after-action (editor)
  (let ((debugger-state (get-debugger-state (serialize-java-object editor))))
    (setf **step-into** t)
    (setf (top-list debugger-state) nil)
    (setf (next-breakpoint debugger-state) nil)
    (stop-after)
    (end-debug-eventloop)))

;;; ***************************************************************************
;;; Called by the interface for the stop-at action
;;; ***************************************************************************

(defun stop-at-action (editor &optional (begin nil) (end nil))
  (let ((debugger-state (get-debugger-state (serialize-java-object editor))))
    (setf **step-into** nil)
    (setf (top-list debugger-state) nil)
    (setf (next-breakpoint debugger-state)
          (if begin
              (list begin end)
            nil))
    (end-debug-eventloop)))

;;; ***************************************************************************
;;; Called by the interface for the continue-execution action
;;; ***************************************************************************

(defun continue-execution-action (editor)
  (let ((debugger-state (get-debugger-state (serialize-java-object editor))))
    (setf (top-list debugger-state) nil)
    (setf (next-breakpoint debugger-state) nil)
    (setf **step-into** nil)
    (end-debug-eventloop)))


;; *****************************************************************
;; Remove all debugging information of a document
;; *****************************************************************

(defun undebug (undebugged-source editor)
  (remhash (serialize-java-object editor) **editors**)
  ;;(setf (gethash editor **breakpoints**) nil)
  (remhash (serialize-java-object editor) **breakpoints**)
  (load undebugged-source))

;; *****************************************************************;;
;; Give control to the interface because we reached breakpoint code ;;
;; Before execution the call
;; **************************************************************** ;;

(defun give-control-to-interface (editor begin end)
  (declare (fixnum begin end))
  ;; Refresh the watches
  (refresh-watches)
  ;; Give control to the interface
  (call-java-method **connection**
                    **interaction**
                    "giveControlToInterface"
                    (deserialize-java-object editor)
                    (1- begin)  ;; In java we count from 0
                    (1- end)))  ;; In java we count from 0


;; ********************************************************************************
;; Display the environment in another time
;; ********************************************************************************

(defun display-time-env (begin end editor)
  (declare (fixnum begin end))
  ;; Refresh the watches
  (refresh-watches)
  ;; Display the new time
  (call-java-method **connection**
                    **interaction**
                    "displayNewTime"
                    (deserialize-java-object editor)
                    (1- begin)
                    (1- end)))

;; *****************************************************************;;
;; Give control to the interface because we reached breakpoint code ;;
;; After execution the call
;; **************************************************************** ;;

(defun give-control-to-interface-after (editor begin end)
  (declare (fixnum begin end))
  ;; Refresh the watches
  (refresh-watches)
  ;; Give control to the interface
  (call-java-method **connection**
                    **interaction**
                    "giveControlToInterfaceAfter"
                    (deserialize-java-object editor)
                    (1- begin) ;; In java we count from 0
                    (1- end))) ;; In java we count from 0

;;; **************************************************************************
;;; Ask interface to display in the watch nodes the new values
;;; **************************************************************************

(defun refresh-watches ()
  (when **watches**
    (call-java-method **connection**
                      "IDE.Watch.Watches"
                      "refreshWatches"
                      (mapcar #'(lambda (n-w)
                                        (list (deserialize-java-object (first n-w))
                                              (eval-expression (second n-w))))
                              **watches**))))


;; ##################################################
;; ##################################################
;; ##################################################
;; ###### SECTION 2 #################################
;; ##################################################
;; ##################################################
;; ##################################################


;; ############################################################################################
;; ############################################################################################
;; #####  READ THE SOURCE INTO A LIST , AND KEEP POSITION INFORMATION IN A HASHTABLE       #### 
;; #####                                                                                   ####
;; ############################################################################################
;; ############################################################################################

;; *********************************************************************************************
;; * read the source and returns a list containig source in reverse and a position hashtable
;; *********************************************************************************************

(defun read-source (source)
  (let ((position (make-hash-table :test #'eq)))
    (values (parse-source (source-to-string source) position) position)))

;; *********************************************************************************************
;; Read the whole source in memory (as one big string)
;; *********************************************************************************************

(defun source-to-string (source)
  (let  ((h-source (open source :direction :input :if-does-not-exist nil)) ;; Open the source
   (line ""))
    (when (null h-source)
      (message (format nil "Can't open ~A") :warn t)
      (return-from source-to-string ""))
    ;; Read to a string and returns the string
    (with-output-to-string (h-str)
         (loop (when (null (setf line (read-line h-source nil nil)))
                         (close h-source)
                         (return))
                          (write-line line h-str)))))



;; ************************************************************************************************
;; Parse the source and produces a list of what was in the source in addition for each list in   **
;; the source a association with a position is saved in the position hashtable                   **
;; Returns the list of parsed elements (or nil in case of parsing error)                         **
;; ************************************************************************************************

(defun parse-source (source position)
  (declare (simple-string source) (hash-table position))
  (let ((i 0) ; index in source
     (l (length source)) ; length of the source
 (frm nil) ; parsed form
        (result nil)) ; result of parsing
    (declare (fixnum i l))
    (catch 'error
      (loop
    (setf i (skip-white source i l))
  (when (>= i l)
    (return result))
      (multiple-value-setq (frm i) (parse-source-element source position i l))
    (push frm result)))))

;;; ****************************************************************************
;;; Function to generate a parse warning, highlight the error position and
;;; throws a 'error
;;; ****************************************************************************

(defun error-parsing (msg i)
  (message msg :warn t)
  (highlight-error i)
  (throw 'error nil))

;;; ****************************************************************************
;;; Use the lisp reader to read the next element , throws a 'error in
;;; case of a parse error
;;; ****************************************************************************

(defun use-lisp-reader (source i)
  (handler-case
      (read-from-string source nil nil :start i)
    (error (er)
     (error (format nil "Parse error {~A}" er) i))))

;;; ****************************************************************************
;;; Dispatcher for the parse of a source element
;;; returns the parsed element and the position after this elements,
;;; throws a 'error if there is a error in parsing 
;;; ****************************************************************************

(defun parse-source-element (source position i l)
  (declare (string source) (hash-table position) (fixnum i l))
  ;; Returns when end of source is reached
  (cond ((>= i l)
   (error-parsing "Try to parse behind end file" i))
      (t
   (case (char source i)
     (#\( ; Start of a list
           (parse-source-element-list source position i l))
    (#\) ; Uncorrect end of a list
          (error-parsing "Found closing hook without opening hook" i))
       (#\" ; Found a string
      (use-lisp-reader source i))
    (#\# ; Found # macro char
       (parse-source-element-hash source position i l))
        (t ; No special character
           (use-lisp-reader source i))))))

;;; ****************************************************************************
;;; Parser for the hash character
;;; ****************************************************************************

(defun parse-source-element-hash (source position i l)
  (declare (string source) (hash-table position) (fixnum i l))
  (incf i) ; skip the hash
  (cond ((>= i l) (error-parsing "Unexpected end of source" i))
        (t (case (char source i)
             (#\'  ; we have a #'
              (incf i)
              (cond ((>= i l) (error-parsing "Unexpected end of source" i))
                    (t (case (char source i)
                         (#\(  ; we have a #'(
                          (let ((frm nil))
                            (multiple-value-setq (frm i)
                                                 (parse-source-element-list
                                                  source
                                                  position
                                                  i
                                                  l))
                            (values (list 'common-lisp::function frm) i)))
                         (t (use-lisp-reader source (- i 2)))))))
             (#\+ (parse-source-element-option t source position i l))
             (#\- (parse-source-element-option nil source position i l))
             (t (use-lisp-reader source (1- i)))))))

;;; ****************************************************************************
;;; Parse of a #+condition form  or #-condition form option (add is t or nil)
;;; Returns (frm i) where i is the next position and
;;; frm is (**option** type condition form) where type is true for #+ and nil
;;; for #-
;;; ****************************************************************************

(defun parse-source-element-option (add source position i l)
  (declare (string source) (hash-table position) (fixnum i l))
  (let ((condition nil)
  (frm nil))
    (multiple-value-setq (condition i) (use-lisp-reader source (1+ i)))
    (multiple-value-setq (frm i) (parse-source-element source position i l))
    (values (list **option** add condition frm) i)))
  

;;; ***************************************************************************
;;; Parse a list, returns (frm i) where frm is the parsed list and i is the
;;; the next position
;;; ***************************************************************************

(defun parse-source-element-list (source position i l)
  (declare (string source) (hash-table position) (fixnum i l))
  (incf i) ; move to position after opening hook
  (let ((frm nil) ; parsed form in list
      (result nil) ; collection of parsed elements in this list
   (begin i)) ; begin position of the hook
    (declare (fixnum begin))
    (loop
      (when (>= i l)
      (error-parsing "Missing closing hook" begin))
      (case (char source i)
   (#\) ; end of the list
    (setf frm (nreverse result)) ; Get the the parsed list
    (setf (gethash frm position) (cons begin (1+ i))) ; Save the begin and end position
      (return (values frm (1+ i))))
     (#\. ; we have a dotted list (a b c ... x . y)
      ;; Get the last element
    (setf i (skip-white source i l))
         (when (>= i l)
          (error-parsing "Missing last element in dotted list" begin))
       (multiple-value-setq (frm i) (parse-source-element source
                                     position
                                 (1+ i)
                                   l))
      ;; Skip eventual whites
    (setf i (skip-white source i l))
         ;; We must have now the closing hook
  (cond ((and (< i l) (char= (char source i) #\)))
           (setf frm (apply #'list* (nreverse (cons frm result))))
         (setf (gethash frm position) (cons begin (1+ i))) ; Save the begin and end position
             (return (values frm (1+ i))))
              (t
       (error-parsing "Missing closing hook in dotted list" begin))))
   (t
        (multiple-value-setq (frm i) (parse-source-element source
                                    position
                                i
                                     l))
    (push frm result)))
      (setf i (skip-white source i l)))))
    
      
;; ******************************************************************************************
;; Parses source , and skips over comments and white spaces                                ;;
;; ******************************************************************************************

(defun skip-white (source i l)
  (declare (string source) (fixnum i l))
  (loop
    (when (>= i l) (return l))
    (case (char source i)
      (#\;
       (loop (when (or (>= i l) (char= (char source i) #\Linefeed)) (return)) (incf i))
       (incf i))
      (#\Space
       (incf i))
      (#\Linefeed
       (incf i))
      (#\Return
       (incf i))
      (#\tab
       (incf i))
      (#\# ; multiline comment #| ..... |#
       (if (and (< (1+ i) l) (char= (char source (1+ i)) #\|))
        (let ((begin i))  
           (loop
         (incf i)
        (when (>= (1+ i) l) (error-parsing "No end of multiline comment"
                                   begin))
               (when (and (char= (char source i) #\|)
                  (char= (char source (1+ i)) #\#))
    (setf i (+ i 2))
            (return))))
        (return i)))
      (t (if (>= i l) (return l) (return i))))))

;; *********************************************************************************************
;; Read in a source , generate a new source where the debugging code is added and read in
;; this new code
;; filename=name of the file containing the source code
;; tempsource=name of the file which should contain the instrumented code
;; editor=the editor buffer being debugged
;; compile=t if we compile the instrumented code             
;; *********************************************************************************************

(defun debug-open-file (filename tempsource editor &optional (compile nil))
  (setf **load** t)
  (let ((ok nil))
    (unwind-protect
     (let ((position (make-hash-table :test #'eq)) 
           (def-lst nil)
           (editor-object (serialize-java-object editor)))
       (db-c-set editor-object position) ; Set the source and position information
       (setf def-lst (parse-source (source-to-string filename) position))
       (cond (def-lst  ; parse succeeded
              (with-open-file (h tempsource :direction :output :if-exists :overwrite)
                              (write
                               ";;; *******************************************"
                               :stream
                               h
                               :escape
                               nil 
                               :readably nil)
                              (terpri h)
                              (write ";;; Generated " :stream h :escape nil :readably nil)
                              (write (get-time) :stream h :escape nil :readably nil)
                              (terpri h)
                              (write
                               ";;; *******************************************"
                               :stream
                               h
                               :escape
                               nil :readably nil)
                              (terpri h)
                              (mapcar
                               #'(lambda (exp)
                                         (write-debugged-exp exp
                                                             position
                                                             editor-object
                                                             h)
                                         (terpri h))
                               (nreverse def-lst)))
              (cond (compile 
                     (compile-file (format nil "~A.lisp" tempsource)
                                   :output-file
                                   tempsource)
                     (load tempsource))
                    (t (load tempsource)))
              (setf ok t)
              ;; Load has succeeded correctly
              ;; Initialize timetravveling to avoid problems with modified sources
              (prepare-time 100))
             (t (setf ok nil))))
     (setf **load** nil))
    (if ok (message "Debugging code added") (message
                                             "Failed to add debugged code"
                                             :warn
                                             t))
    ok))
  
;;; ****************************************************************************
;;; Write the debugged code , two cases
;;; 1. exp is of the form (**option** add cond form) must be written as
;;;    #+cond form or #-cond form (depending if add is + or -
;;; 2. exp is a form 
;;; ****************************************************************************

(defun write-debugged-exp (exp position editor h)
  (cond ((listp exp)
         (cond ((eq (first exp) **option**) ; Option list
                (let ((add (second exp))
                      (cnd (third exp))
                      (frm (fourth exp)))
                  (if add
                      (write "#+" :stream h :escape nil :readably nil :radix nil :base 10 :pretty nil
                             :level nil :length nil :gensym nil :array nil)
                    (write "#-" :stream h :escape nil :readably nil :radix nil :base 10 :pretty nil
                           :level nil :length nil :gensym nil :array nil))
                  (write cnd :stream h :escape t :radix nil :base 10 :pretty t
                         :level nil :length nil :gensym nil :array t)
                  (terpri h) 
                  (write-debugged-exp-list (add-debugging-code frm position editor) h)))
               (t ; Not a option list
                (write-debugged-exp-list (add-debugging-code exp position editor) h))))
        (t
         (write exp :stream h :escape t :radix nil :base 10 :pretty t
                :level nil :length nil :gensym nil :array t))))

(defun write-debugged-exp-list (exp h)
  (write #\( :stream h :escape nil :readably nil)
  (loop (cond ((null exp) (return))
              ((atom exp) 
               (write #\space :stream h :escape nil :readably nil)
               (write #\. :stream h :escape nil :readably nil)
               (write #\space :stream h :escape nil :readably nil)
               (write-debugged-exp-list-element exp h)
               (return))
              (t 
               (write-debugged-exp-list-element (first exp) h)
               (pop exp)
               (when (not (atom exp)) (write #\space :stream h :escape nil :readably nil)))))
  (write #\) :stream h :escape nil :readably nil)
  (terpri h))

(defun write-debugged-exp-list-element (exp h)
  (cond ((listp exp)
         (cond ((eq (first exp) **option**) ; Option list
                (let ((add (second exp))
                      (cnd (third exp))
                      (frm (fourth exp)))
                  (if add
                      (write "#+" :stream h :escape nil :readably nil :radix nil :base 10 :pretty nil
                             :level nil :length nil :gensym nil :array nil)
                    (write "#-" :stream h :escape nil :readably nil :radix nil :base 10 :pretty nil
                           :level nil :length nil :gensym nil :array nil))
                  (write cnd :stream h :escape t :radix nil :base 10 :pretty t
                         :level nil :length nil :gensym nil :array t)
                  (terpri h) 
                  (write-debugged-exp-list frm h)))
               (t ; Not a option list
                (write-debugged-exp-list exp h))))
        (t
         (write exp :stream h :escape t :radix nil :base 10 :pretty t
                :level nil :length nil :gensym nil :array t))))
     
   
;; ##################################################
;; ##################################################
;; ##################################################
;; ###### SECTION 3 #################################
;; ##################################################
;; ##################################################
;; ##################################################

;; ******************************************************************************
;; ******************************************************************************
;; Code to generate new code , which transforms a source in a source with 
;; debugcode added 
;; ******************************************************************************
;; ******************************************************************************

;; ************************************************************************************************************
;; Parse a text file of and generate a list of definitions
;;
;; Syntax ::
;;
;; def :==: symbol -- = --- ( -- def-exp ... def-exp -- ) --- "Subdefinition"
;;     :==: --- ( --- symbol def-exp ... def-exp --- ) ---    "Definition"
;;     :==: "symbol"                                          "Do in package to symbol"
;; def-exp :==: symbol                                        "Symbol , piece of text"
;;         :==: _symbol                                       "Represents any lisp expression"
;;         :==: ~symbol                                       "Represents a variable"
;;         :==: #symbol                                       "Represents any lisp expression where debug code must be added"
;;         :==: @                                             "Represents begin usage of variable of type 2 , and add of debugpoint"
;;         :==: "symbol"                                      "Represents a string"
;;         :==: --- [ --- def-exp .... def-exp --- ] ---      "Optional expressions"
;;         :==: --- [ --- def-exp .... def-exp --- ] --- * -- "Expressions" occuring zero or more times"
;;         :==: --- { --- def-exp | ... | def-exp --- } ---   "Different possible expressions"
;;         :==: -- { -- def-exp | ... | def-exp -- } -- * --  "zero or more possible expressions"
;; **********************************************************************************************
;; Using the meaning of the previous definitions we must translate the code as follows :
;; **********************************************************************************************
;; symbol ---> symbol
;; _symbol ---> **var**
;; ~symbol ---> **variable2**
;; #symbol ---> **debug**
;; "symbol" ---> **string**
;; @       ---> **debug-first**
;; [ def-exp ... def-exp ] ---> (**try** -- ....)
;; [ def-exp ... def-exp ] ---> (**try** -- ....)
;; [ def-exp ... def-exp ] * ---> (**repeat** ....)
;; { def-exp } ---> translated def-exp
;; { def-exp } * ---> tranlated def-exp + (**repeat** tranlated def-exp)
;; { def-exp | ... def-exp } ---> (**or** translated exp ....)
;; { def-exp | ... def-exp } * --> (**repeat** (**or** ...))
;; Remark: Escape characters to be able to use characters like [,],{,},|,*,,_,~,#,",@,\ and = in symbols
;;         we use the \ to indicate that this character is actually part of a symbol and not a 
;;         escape character
;; ************************************************************************************************

;; *****************************************
;; Parameters used to indicate certain 
;; symbols
;; *****************************************

(defparameter **variable2** (gensym))
(defparameter **var** (gensym))
(defparameter **debug** (gensym))
(defparameter **debug-first** (gensym))
(defparameter **use-var** (gensym))
(defparameter **or** (gensym))
(defparameter **try** (gensym))
(defparameter **repeat** (gensym))
(defparameter **string** (gensym))
(defparameter **undefined** (gensym))


;; *****************************************
;; Parse of a string containing syntax 
;; definitions 
;; *****************************************


(defun parse-definition-start (source subdef)
  (declare (simple-string source))
  (parse-definition source 0 (length source) subdef))

(defun parse-definition (source i l subdef)
  (declare (string source) (fixnum i l))
  (let ((result nil)
      (def nil))
    (loop
      (setf i (skip-white source i l))
      (when (>= i l) (return (nreverse result)))
      (case (char source i)
     (#\"
        (setf i (parse-definition-package source (1+ i) l)))
        (#\(
   (multiple-value-setq (def i) (parse-definition-list source i l))
        (push def result))
  (t
       (setf i (parse-definition-subdef source i l subdef)))))))
#-CLISP
(defun parse-definition-package (source i l)
  (declare (string source) (fixnum i l))
  (let ((symbol ""))
    (declare (string symbol))
    (multiple-value-setq (symbol i) (parse-definition-string source i l))
    (when (not (string= symbol ""))
      (when (not (find-package symbol)) (make-package symbol))
      (in-package symbol))
    i))
;; There is a bug in the in-package implementation causing it to be evaluated
;; even when not a top level form
#+CLISP
(defun parse-definition-package (source i l)
  (declare (string source) (fixnum i l))
  (let ((symbol ""))
    (declare (string symbol))
    (multiple-value-setq (symbol i) (parse-definition-string source i l))
    (when (not (string= symbol ""))
      (when (not (find-package symbol)) (make-package symbol))
      (eval (list 'in-package symbol)))
    i))

(defun parse-definition-string (source i l)
  (declare (string source) (fixnum i l))
  (values (with-output-to-string (h)
                           (loop (cond ((>= i l)
                     (message "Missing \"" :warn t)
                           (throw 'error nil)
                     (return))
                                   ((char= (char source i) #\")
                                 (return))
                       ((is-break-char (char source i))
                                 (message "Invalid char found in string" :warn t)
                     (throw 'error i)
                                 (return))
                       (t
                               (princ (char source i) h)
                              (incf i)))))
    (1+ i)))


(defun is-break-char (c)
  (or (char= c #\=)
      (char= c #\()
      (char= c #\))
      (char= c #\")
      (char= c #\_)
      (char= c #\~)
      (char= c #\#)
      (char= c #\@)
      (char= c #\[)
      (char= c #\])
      (char= c #\{)
      (char= c #\})
      (char= c #\|)
      (char= c #\;)
      (char= c #\space)
      (char= c #\return)
      (char= c #\tab)
      (char= c #\newline)))


(defun parse-definition-subdef (source i l subdef)
  (declare (string source) (fixnum i l))
  (let ((symbol nil)
     (def nil))
    (multiple-value-setq (symbol i) (parse-definition-symbol source i l))
    (when (gethash symbol subdef)
      (message "Subdefinition exist already" :warn t)
      (throw 'error i))
    (setf i (skip-white source i l))
    (when (or (>= i l) (not (char= (char source i) #\=)))
      (message "Missing =" :warn t)
      (throw 'error i))
    (setf i (skip-white source (1+ i) l))
    (multiple-value-setq (def i) (parse-definition-exp source  i l))
    (setf (gethash symbol subdef) def)
    i))


(defun parse-definition-symbol (source i l)
  (declare (string source) (fixnum i l))
  (when (is-break-char (char source i)) 
    ;; Avoid looping on breakchar
    (loop (cond ((or (>= i l) (not (is-break-char (char source i)))) (return))
                (t (incf i)))))
  (setf i (skip-white source i l))
  (values (read-from-string (with-output-to-string (h)
                                                   (loop
                                                    (cond ((>= i l) (return))
                                                          ((char= #\\ 
                                                                  (char source i))
                                                           (incf i)
                                                           (when (< i l)
                                                             (princ (char source
                                                                          i)
                                                                    h)))
                                                          ((is-break-char
                                                            (char source i)) 
                                                           (return))
                                                          (t (princ (char source
                                                                          i)
                                                                    h)))
                                                    (incf i))))
          i))
                         

(defun parse-definition-list (source i l)
  (declare (string source) (fixnum i l))
  (incf i) ; skip the opening hook
  (let ((exp nil)
       (result nil))
    (loop
      (setf i (skip-white source i l))
      (when (>= i l)
  (message "Missing )" :warn t)
   (throw 'error nil))
      (case (char source i)
  (#\) ; End of list encountered
   (return (values (nreverse result) (1+ i))))
     (#\. ; We have a dotted list
        ; Read the last element
      (setf i (skip-white source (1+ i) l))
      (when (>= i l)
       (message "Missing last element in dotted list" :warn t))
        (multiple-value-setq (exp i) (parse-definition-exp source i l))
      ; We must have now only a ) left
   (setf i (skip-white source i l))
        (cond ((char= (char source i) #\))
   (return (values (apply #'list* (nreverse (cons exp result)))
          (1+ i))))
             (t
     (message "Missing ) in dotted list" :warn t))))
 (t (multiple-value-setq (exp i) (parse-definition-exp source i l))
        (push exp result))))))


(defun parse-definition-exp (source i l)
  (declare (string source) (fixnum i l))
  (let ((exp nil))
    (when (>= i  l) (return-from parse-definition-exp (values nil i)))
    (case (char source i)
      (#\(
       (parse-definition-list source i l))
      (#\)
       (message "Unexpeced ) encountered" :warn t)
       (throw 'error i))
      (#\_  ;; **var**
       (multiple-value-setq (exp i) (parse-definition-symbol source (1+ i) l))
       (values **var** i))
      (#\~ ;; **variable2**
       (multiple-value-setq (exp i) (parse-definition-symbol source (1+ i) l))
       (values **variable2** i))
      (#\# ;; **debug**
       (multiple-value-setq (exp i) (parse-definition-symbol source (1+ i) l))
       (values **debug** i))
      (#\"
       (multiple-value-setq (exp i) (parse-definition-string source (1+ i) l))
       (values **string** i))
      (#\@ ;; **debug-first**
       (values **debug-first** (1+ i)))
      (#\[
       (parse-definition-try source (1+ i) l))
      (#\]
       (message "Unexpected ] found" :warn t)
       (throw 'error i))
      (#\{
       (parse-definition-or source (1+ i) l))
      (#\}
       (message "Unexpected } found" :warn t)
       (throw 'error i))
      (#\|
       (parse-definition-exp source (+ i 1) l))
      (#\*
       (parse-definition-exp source (+ i 1) l))
      (t
       (parse-definition-symbol source i l)))))




(defun parse-definition-try (source i l)
  (declare (string source) (fixnum i l))
  (let ((exp nil)
        (result nil))
    (loop
      (setf i (skip-white source i l))
      (when (>= i l)
   (message "] missing" :warn t)
    (throw 'error nil))
      (case (char source i)
   (#\[
      (multiple-value-setq (exp i) (parse-definition-try source (1+ i) l))
       (push exp result))
 (#\]
    (setf i (skip-white source (1+ i) l))
    (cond ((or (>= i l) (not (char= (char source i) #\*)))
   (return (values (cons **try** (nreverse result)) i)))
          (t
               (return (values (cons **repeat** (nreverse result)) (1+ i))))))
       (t
    (multiple-value-setq (exp i) (parse-definition-exp source i l))
  (push exp result))))))

      

(defun parse-definition-or (source i l)
  (declare (string source) (fixnum i l))
  (let ((exp nil)
    (result nil))
    (loop
      (setf i (skip-white source i l))
      (when (>= i l)
       (message "} missing" :warn t)
        (throw 'error nil))
      (case (char source i)
       (#\}
  (setf i (skip-white source (1+ i) l))
  (cond ((or (>= i l) (not (char= (char source i) #\*)))
               (return (values (cons **or** (nreverse result)) i)))
         (t
             (if (endp (rest result))
           (return (values (cons **repeat** (first result)) (1+ i)))
         (return (values (list **repeat** (cons **or** (nreverse result))) (1+ i)))))))
  (t
       (multiple-value-setq (exp i) (parse-definition-or-choice source i l))
       (when (not (null exp))
        (push exp result)))))))


(defun parse-definition-or-choice (source i l)
  (declare (string source) (fixnum i l))
  (let ((result nil)
  (exp nil))
  (loop
    (setf i (skip-white source i l))
    (when (>= i l)
      (return (values (nreverse result) i)))
    (case (char source i)
      (#\}
       (return (values (nreverse result) i)))
      (#\|
       (return (values (nreverse result) (1+ i))))
      (t
       (multiple-value-setq (exp i) (parse-definition-exp source i l))
       (push exp result))))))




;; *******************************************************************************
;; Code to analyse the structure of a expression and generate a new expression
;; *******************************************************************************
;;
;; The principe is as follows :
;;
;; 1. We start with a fle containing syntax expressions , defining lisp expressions ,and 
;;    indicating how debug code must be added
;; 2. This file is parsed , generating code in a intermediate language , defining lisp expressions
;; 3. This intermediate code is transformed in real lisp functions which transform a 
;;    lisp expression in a lispexpression whith debugging code added
;; **********************************************************************************


;; **********************************************************************************
;; First we define functions who generate lisp functions of the following type
;; (lambda (exp acc rem) ...) ---> exp',acc' 
;; Where : exp,acc is transformed in exp',acc' by the code of the function 
;;         If in the code an error is encoutered a throw to nomatch is done.
;;
;; Remark : If rem is not nil , a continuation call of the form (funcall (first rem) exp' acc' (rest rem))
;;          is done 
;; **********************************************************************************

;; **********************************************************************************
;; Used for dotted list's (to process last element in the dotted list
;; (x1   xn-1 . xn)
;; (db-after-dot f1) ==> f with
;; (f exp acc rem) ==> nil,acc' where (f1 (exp) nil rem) == > acc'
;; **********************************************************************************

(defun db-after-dot (fun)
  #'(lambda (exp acc rem)
      (cond ((not (atom exp))
           (throw 'nomatch nil))
      (t
       (funcall fun (list exp) acc rem)))))

;; (setf f (db-after-dot (db-constant 'a)))
;; (funcall f 'a nil nil)

;; **********************************************************************************
;; (db-constant 'a) ==> f with
;; (f '(a | r) acc nil) ==> r , (a | acc),nil
;; **********************************************************************************

(defun db-constant (constant)
  #'(lambda (exp acc rem)
      (cond ((atom exp) 
         (throw 'nomatch nil))
            ((eql (first exp) constant)
             (cond ((endp rem)
                   (values (rest exp) (cons (first exp) acc)))
              (t
               (funcall (first rem) (rest exp) (cons (first exp) acc) (rest rem)))))
      (t
       (throw 'nomatch nil)))))

;; 
;; Tests
;;
;; (setf f (db-constant 'a))
;; (funcall f '(a b c) nil nil)
;; (funcall f '(a n c) nil (list (db-constant 'n)))
;; (funcall f '(b c a) nil nil)
;;


;; *********************************************************************************
;; (db-sequence f1) ==> f with 
;; (f exp acc rem) ==> (f1 exp acc rem)
;; (db-sequence f1 ... fn) ==> f with
;; (f exp acc rem) == (f1 exp acc ((f2 ...fn | rem)
;; Remark : (db-sequence) is an illegal call
;; *********************************************************************************

(defun db-sequence (&rest fun-lst)
  #'(lambda (exp acc rem)
      (cond ((endp (rest fun-lst))
       (funcall (first fun-lst) exp acc rem))
         (t
          (funcall (first fun-lst) exp acc (append (rest fun-lst) rem))))))

;; (setf f (db-sequence (db-constant 'a) (db-constant 'b)))
;; (funcall f '(a b) nil nil)
;; (funcall f '(a b c) nil (list (db-constant 'c)))
;; (funcall f '(a c) nil nil)



;; **********************************************************************************
;; (db-try try rst) ==> f where we try the following , until we have no throw (if 
;; possible)
;; old 1.  (rst exp acc rem) ==> exp' ,acc'
;; old 2.  (try exp acc (rst | rem)) ==> exp' ,acc'
;; 1.  (try exp acc (rst | rem)) == exp' , acc'
;; 2.  (rst exp acc rem) ==> exp' , acc'
;; *********************************************************************************


(defun db-try (try rst)
  #'(lambda (exp acc rem)
      (let ((n-exp nil)
         (n-acc nil)
        (save nil))
   (cond ((catch 'nomatch
      (setf save (db-c-get))
          (multiple-value-setq (n-exp n-acc) (funcall try exp acc (cons rst rem)))
          t)
              (values n-exp n-acc))
       (t
              (db-c-put save)
              (funcall rst exp acc rem))))))

;; (setf f (db-try (db-constant 'a) (db-constant 'b)))
;; (funcall f '(b) nil nil)
;; (funcall f '(a b) nil nil)
;; (funcall f '(a b c) nil nil)
;; (funcall f '(c b) nil nil)
;; (funcall f '(a c) nil nil)


;; *********************************************************************************
;; (db-repeat try rst) ==> f where we try the following , until we have no throw 
;; (if possible)
;; old 1. (rst exp acc rem) ==> exp',acc'
;; old 2. (try exp acc (rst | rem))
;; old 3. (try exp acc (try | (rst |rem))
;;      ...
;; We try as many time as possible to execute try before we execute rst
;; So essentially we do
;; 1. (try exp acc (try | try .... | rem)) avoiding infinite looping
;; 2. (rst exp acc rem)                             
;; *********************************************************************************


(defun db-repeat (try rst)
  #'(lambda (exp acc rem)
      (let ((self nil))
      (setf self #'(lambda (e a r)
                      (let ((save (db-c-get))
                       (n-exp nil)
           (n-acc nil))
                (cond ((atom e)
                  (funcall rst e a r))
                       ((catch 'nomatch
                    (multiple-value-setq (n-exp n-acc)
                  (funcall try e a (cons self r)))
            t)
              (values n-exp n-acc))
                       (t
          (db-c-put save)
                          (funcall rst e a r))))))
   (funcall self exp acc rem))))

   
;; (setf f (db-repeat (db-constant 'a) (db-constant 'b)))
;;
;; (funcall f '(b) nil nil)
;; (funcall f '(a b) nil nil)
;; (funcall f '(a a b) nil nil)
;; (funcall f '(a a a b) nil nil)
;; (funcall f '(a c) nil nil)
;; (funcall f '(c) nil nil)
;; 
;; (setf f (db-repeat (db-sequence (db-repeat (db-constant 'a) #'db-c-nothing) (db-constant 'b)) (db-constant 'c)))
;; (funcall f '(c) nil nil)
;; (funcall f '(b c) nil nil)
;; (funcall f '(a b c) nil nil)
;; (funcall f '(a a b c) nil nil)
;; (funcall f '(a a b a a b c) nil nil)
;; (funcall f '(a a b a a b d) nil nil)
;;
            
;; *********************************************************************************
;; (db-or '(f1 ... fn) fun) ==> f with
;;
;; (f exp acc rem) ==> exp',acc' with exp',acc' the result of the first non 
;; throwing possibilities
;;
;; 1. (f1 exp acc (fun | rem))
;;         ....
;; n. (f1 exp acc *fun | rem))
;; *********************************************************************************


(defun db-or (or-lst fun)
  #'(lambda (exp acc rem)
      (let ((n-exp nil)
          (n-acc nil)
         (n-or-lst or-lst)
          (save (db-c-get)))
      (loop
         (db-c-put save)
      (cond ((endp n-or-lst)
    (throw 'nomatch nil))
         ((catch 'nomatch 
            (multiple-value-setq (n-exp n-acc) (funcall (first n-or-lst) exp acc (cons fun rem)))
      t)
    (return (values n-exp n-acc)))
               (t
      (pop n-or-lst)))))))

;;
;; (setf f (db-or (list (db-constant 'x) (db-constant 'y) (db-constant 'z)) (db-sequence (db-constant 'b) (db-constant 'c))))
;; (funcall f '(x b c) nil nil)
;; (funcall f '(y b c) nil nil)
;; (funcall f '(z b c) nil nil)
;;

;; **********************************************************************************
;; (db-list fun) ==> f with
;; (f exp acc rem) ==> exp , acc where we have 
;; (db-c-push (rest exp) ((reverse res) | acc) (#'db-c-pop | #'db-c-end | rem))
;; and we have res generated by 
;; (fun (first exp) nil (#'db-c-end))
;; **********************************************************************************


(defun db-list (fun)
  #'(lambda (exp acc rem)
      (let ((n-exp nil)
         (n-acc nil))
   (cond ((not (listp (first exp)))
        (throw 'nomatch nil))
         ((catch 'nomatch
      (db-c-push (first exp))
                (multiple-value-setq (n-exp n-acc) (funcall fun (first exp) nil (list #'db-c-end)))
          t)
          (db-c-pop)
               (cond ((endp rem)
                 (values (rest exp) (cons (nreverse n-acc) acc)))
                (t
             (funcall (first rem) (rest exp) (cons (nreverse n-acc) acc) (rest rem)))))
        (t 
              (db-c-pop)
           (throw 'nomatch nil))))))

;; **********************************************************************************
;; (db-dotted-list fun) ==> f with
;; (f exp acc rem) ==> exp , acc where we have 
;; (db-c-push (rest exp) ((apply #'list* (reverse res)) | acc) (#'db-c-pop | #'db-c-end | rem))
;; and we have res generated by 
;; (fun (first exp) nil (#'db-c-end))
;; **********************************************************************************


(defun db-dotted-list (fun)
  #'(lambda (exp acc rem)
      (let ((n-exp nil)
       (n-acc nil))
 (cond ((not (listp (first exp)))
              (throw 'nomatch nil))
       ((catch 'nomatch
          (db-c-push (first exp))
            (multiple-value-setq (n-exp n-acc)
                  (funcall fun (first exp) nil nil))
                t)
         (db-c-pop)
              (cond ((endp rem)
               (values (rest exp)
                 (cons (apply #'list* (nreverse n-acc)) acc)))
            (t
         (funcall (first rem)
                   (rest exp)
                            (cons (apply #'list* (nreverse n-acc)) acc)
             (rest rem)))))
        (t
               (db-c-pop)
            (throw 'nomatch nil))))))

;;
;; (setf f (db-dotted-list (db-sequence (db-constant 'a) (db-constant 'b) (db-after-dot (db-constant 'c)))))
;;
;; (funcall f '((a b . c) c) nil nil)
;; (funcall f '((a b c) c) nil nil)
;;



;; **********************************************************************************
;; (db-call-exp fun) ==> f of the form (lambda (exp) ...) ==> exp'
;; (f exp) ==> 'exp , with (funcall fun exp nil nil) === e,v and 'exp=reverse(v) 
;;             nil is returned in case of a throw
;; **********************************************************************************

(defun db-call-exp (fun)
  #'(lambda (exp)
      (let ((n-exp nil)
            (n-acc nil))
      (cond ((catch 'nomatch
            (db-c-push exp)
      (multiple-value-setq (n-exp n-acc) (funcall fun exp nil (list #'db-c-end)))
        t)
             (db-c-pop)
          (nreverse n-acc))
        (t
               (db-c-pop)
            nil)))))

;; **********************************************************************************
;; Already generated functions
;; **********************************************************************************


;; **********************************************************************************
;; Called when we want to add debug code 
;; **********************************************************************************

(defun db-c-debug (exp acc rem)
  (cond ((atom exp)
    (throw 'nomatch nil))
   ((endp rem)
       (values (rest exp) (cons (add-debug-point (first exp) (db-c-variable) (db-c-position) (db-c-source)) acc)))
        (t
     (funcall (first rem)
           (rest exp)
             (cons (add-debug-point (first exp) (db-c-variable) (db-c-position) (db-c-source)) acc)
       (rest rem)))))


;; **********************************************************************************
;; Checks if we have detected the end of exp
;; **********************************************************************************

(defun db-c-end (exp acc rem)
  (cond ((and (null exp) (null rem))
  (values nil acc))
     (t
  (throw 'nomatch nil))))

;; (funcall #'db-c-end nil nil nil)
;; (funcall #'db-c-end '(a) nil nil)
;; (funcall #'db-c-end '(a) nil (list (db-constant 'a)))


;; **********************************************************************************
;; Do nothing , except calling the continuation code
;; **********************************************************************************

(defun db-c-nothing (exp acc rem)
  (cond ((endp rem)
   (values exp acc))
      (t
   (funcall (first rem) exp acc (rest rem)))))


;; **********************************************************************************
;; (db-c-var exp acc nil) ==> 
;;
;; 1. throw , if exp is endp
;; 2. (rest exp), (cons (first exp) acc)
;; **********************************************************************************

(defun db-c-var (exp acc rem)
  (cond ((atom exp)
    (throw 'nomatch nil))
   ((endp rem)
       (values (rest exp) (cons (first exp) acc)))
        (t
     (funcall (first rem) (rest exp) (cons (first exp) acc) (rest rem)))))

;; **********************************************************************************
;; (db-c-empty-list exp acc nil) ==>
;; 1. throw , if exp is atom or (first exp) is not a empty list
;; 2. (rest exp) , (cons (first exp) acc)
;; **********************************************************************************

(defun db-c-empty-list (exp acc rem)
  (cond ((atom exp)
       (throw 'nomatch nil))
      ((not (null (first exp)))
    (throw 'nomatch nil))
   ((endp rem)
       (values (rest exp) (cons (first exp) acc)))
        (t
     (funcall (first rem) (rest exp) (cons (first exp) acc) (rest rem)))))

;; **********************************************************************************
;; (db-c-string exp acc nil) ==> 
;;
;; 1. throw , if exp is endp or (first exp) is not a string
;; 2. (rest exp), (cons (first exp) acc)
;; **********************************************************************************

(defun db-c-string (exp acc rem)
  (cond ((atom exp)
        (throw 'nomatch nil))
       ((not (stringp (first exp)))
  (throw 'nomatch nil))
 ((endp rem)
     (values (rest exp) (cons (first exp) acc)))
      (t
   (funcall (first rem) (rest exp) (cons (first exp) acc) (rest rem)))))


;; **********************************************************************************
;; Functions to handle lexical definitions of variables in a function
;; **********************************************************************************

(let ((expression-stack nil)
      (variable-stack nil)
      (variable nil)
      (source nil)
      (expression nil)
      (position nil))
  (defun db-c-get ()
    (list expression-stack variable-stack variable expression))
  (defun db-c-put (lst)
    (setf expression-stack (pop lst))
    (setf variable-stack (pop lst))
    (setf variable (pop lst))
    (setf expression (pop lst)))
  ;; Sets position and expression info
  (defun db-c-set (src pos)
    (setf position pos)
    (setf source src))
  (defun db-c-clear ()
    (setf variable-stack nil)
    (setf expression-stack nil)
    (setf variable nil))
  ;; Gets position info
  (defun db-c-position ()
    position)
  ;; Gets source info
  (defun db-c-source ()
    source)
  ;; Gets expression info
  (defun db-c-expression ()
    expression)
  ;; Gets variable info
  (defun db-c-variable ()
    variable)
  ;; Push expression on stack
  (defun db-c-set-exp (exp)
    (setf expression exp))
  ;; push variable info on the stack , to be used in the begin of a list
  (defun db-c-push (exp)
    (push expression expression-stack)
    (setf expression exp)
    (push variable variable-stack))
  ;; pop variable information from the stack , to be used on the end of a list
  (defun db-c-pop ()
    (setf expression (pop expression-stack)))
  ;; activate variables at a chosen moment + add debugging code 
  (defun db-c-use-var (exp acc rem)
    (cond ((endp rem)
    (values exp (cons (add-debug-point-first (db-c-expression) (db-c-variable) (db-c-position) (db-c-source)) acc) rem))
          (t
     (funcall (first rem) exp (cons (add-debug-point-first (db-c-expression) (db-c-variable) (db-c-position) (db-c-source)) acc) (rest rem)))))
  ;; Append (first exp) to variable-2
  (defun db-c-variable-2 (exp acc rem)
    (cond ((or (atom exp)
           (keywordp (first exp))
            (member (first exp) '(&allow-other-keys &aux &body &environment &key &optional &rest &whole))) 
        (throw 'nomatch nil))
     ((listp (first exp))
      (throw 'nomatch nil))
   ((endp rem)
     (pushnew (first exp) variable)
          (values (rest exp) (cons (first exp) acc)))
         (t
    (pushnew (first exp) variable)
         (funcall (first rem) (rest exp) (cons (first exp) acc) (rest rem)))))
  )
    


;; ************************************************************************************
;; Translation of parsed version of the syntaxfile using the previous defined 
;; code generators
;; ************************************************************************************
;;
;; After parsing of the syntax file we have expressions like :
;;
;;
;; definition :==: -- (  -- definition-exp ... definition-exp -- ) --
;; definition-exp :==: definition
;; definition-exp :==: **string**
;; definition-exp :==: symbol
;; definition-exp :==: **var**
;; definition-exp :==: **variable2**  (variable defined in let , do ...
;; definition-exp :==: **debug**
;; definition-exp :==: (function-s symbol)
;; definition-exp :==: **debug-first** pseudo code added to show whole expression
;;                     becomes active and first call of master expression
;; definition-exp :==: -- ( -- **try** -- definition-exp ... definition-exp -- ) --
;; definition-exp :==: -- ( -- **repeat** -- definition-exp -- ) --
;; definition-exp :==: -- ( -- **or** -- definition-exp ... definition-exp --- ) ---
;;
;; ************************************************************************************
;; 
;; Transformation transform this code in the following way
;;
;; (tr **string** nil) ==> (#'db-c-string)
;; (tr **string** r)   ==> (#'db-c-string | (tr (first r) (rest r)))
;; (tr symbol nil)  ==> ((db-constant 'symbol))
;; (tr symbol r)    ==> ((db-constant 'symbol) | (tr (first r) (rest r)))
;; (tr **var** nil) ==> (#'db-c-var)
;; (tr **var** r)   ==> (#'db-c-var | (tr (first r) (rest r)))
;; (tr **variable2** nil) ==> (#'db-c-variable2)
;; (tr **variable2** r)   ==> (#'db-c-variable2 | (tr (first r) (rest r)))
;; (tr **debug** nil) ==> (#'db-c-debug)
;; (tr **debug** r)   ==> (#'db-c-debug | (tr (first r) (rest r)))
;; (tr **debug-first** nil) ==> (#'db-c-use-var)
;; (tr **debug-first** r)   ==> (#'db-c-use-var | (tr (first r) (rest r)))
;; (tr (a1 ... an) nil) ==> ((db-list (tr-sequence (tr a1 (a2 .. an)))))
;; (tr (a1 ... an) r)   ==> ((db-list (tr-sequence (tr a1 (a2 ...an)))) | (tr (first r) (rest r)))
;; (tr (function-s symbol) nil) ==> ((function symbol))
;; (tr (function-s symbol) r)   ==> ((function symbol) | (tr (first r) (rest r)))
;; (tr (**repeat** a1 ...an) nil)    ==> ((db-repeat (tr-sequence (tr a1 (a2 ..an))) #'db-c-nothing)) (1)
;;                                   ==> ((db-repeat-simple (tr-sequence (tr a1 (a2 .. an))) #'db-c-nothing)) (2)
;; (tr (**repeat** a1 ...an) r)    ==> ((db-repeat (tr-sequence (tr a1 (a2 ..an))) (tr-sequence (tr (first r) (rest r))))) (1)
;;                                 ==> ((db-repeat-simple (tr-sequence (tr a1 (a2 .. an))) (tr-sequence (tr (first r) (rest r))))) (2)
;; (tr (**try** a1 .. an) nil) ==> ((db-try (tr-sequence (tr a1 (tr a2 ..an))) #'db-c-nothing))
;; (tr (**try** a1 ...an) r)   ==> ((db-try (tr-sequence (tr a1 (tr a2 ..an))) (tr-sequence (tr (first r) (rest r))))
;; (tr (**or** l1 ... ln) nil) ==> ((db-or ((tr-sequence (tr l1 nil)) ... (tr-sequence ln nil))) #'db-c-nothing))
;; (tr (**or** l1 ... ln) nil) ==> ((db-or ((tr-sequence (tr (first l1) (rest l1))) ... (tr-sequence (tr (first ln) (rest ln))))
;;                                         (tr-sequence (tr (first r) (rest r)))))
;;
;; (tr-sequence (a)) ==> a
;; (tr-sequence (a1 ... an)) ==> (db-sequence a1 ... an)
;;
;; (1) (tr-simple (a1 ...an)) is T
;; (2) (tr-simple (a1 ...an)) is nil
;;
;; (tr-simple (**repeat** ...)) ==> nil
;; (tr-simple (**try** ...)) ==> nil
;; (tr-simple (a1 ... an)) ==> (every #'tr-simple (a1 ... an)) (should be modified see to-java-simple)
;; (tr-simple x) ==> T
;;
;; ******************************************************************************************************************************************
;; Previous transformation function is used to generate the following functions
;; ******************************************************************************************************************************************
;; function to represent a named syntax rule 
;;
;; (tr-named-rule-1 name syntax-exp) ==> (setf (symbol-function 'name) (tr-sequence (tr (first syntax-exp) (rest syntax-exp))))
;; (tr-nemae-rule-2 name syntax-exp) ==> (defun name (exp acc rem)
;;                                          (funcall (tr-sequence (tr (first syntax-exp) (rest syntax-exp))) exp acc rem))
;;
;; function to represent a non named syntax rule
;;
;; (tr-rule name syntax-exp) ==> (setf (symbol-function 'name) (db-call-exp (tr-seqeunce (tr (first syntax-exp) (rest syntax-exp)))))
;; ******************************************************************************************************************************************


(defun tr-sequence (lst)
  (cond ((endp (rest lst))
  (first lst))
  (t (cons 'db-sequence lst))))

(defun tr-dotted-sequence (lst)
  (cons 'db-sequence (tr-dotted-sequence-1 lst)))

(defun tr-dotted-sequence-1 (lst)
  (cond ((endp (rest lst))
   (list (list 'db-after-dot (first lst))))
       (t (cons (first lst) (tr-dotted-sequence-1 (rest lst))))))

(defun tr (frst rst)
  (cond ((listp frst)
        (cond ((eq (first frst) **repeat**)
         (tr-repeat frst rst))
        ((eq (first frst) **try**)
           (tr-try frst rst))
        ((eq (first frst) **or**)
             (tr-or frst rst))
          ((proper-list-p frst)
         (tr-list frst rst))
              (t
       (tr-dotted-list frst rst))))
     ((function-s-p frst)
        (tr-function frst rst))
     ((eq frst **string**)
       (tr-string frst rst))
      ((eq frst **var**)
   (tr-var frst rst))
     ((eq frst **variable2**)
    (tr-variable-2 frst rst))
       ((eq frst **debug**)
  (tr-debug frst rst))
  ((eq frst **debug-first**)
       (tr-first-debug frst rst))
 (t
      (tr-symbol frst rst))))

(defun tr-string (frst rst)
  (cond ((atom rst)
   (list (list 'function 'db-c-string)))
  (t 
      (cons (list 'function 'db-c-string)
              (tr (first rst) (rest rst))))))


(defun tr-symbol (frst rst)
  (cond ((atom rst)
      (list (list 'db-constant (list 'quote frst))))
    (t 
        (cons (list 'db-constant (list 'quote frst))
               (tr (first rst) (rest rst))))))


(defun tr-var (frst rst)
  (cond ((atom rst)
  (list (list 'function 'db-c-var)))
    (t 
        (cons (list 'function 'db-c-var)
           (tr (first rst) (rest rst))))))



(defun tr-variable-2 (frst rst)
  (cond ((atom rst)
      (list (list 'function 'db-c-variable-2)))
 (t 
     (cons (list 'function 'db-c-variable-2)
         (tr (first rst) (rest rst))))))


(defun tr-debug (frst rst)
  (cond ((atom rst)
  (list (list 'function 'db-c-debug)))
  (t 
      (cons (list 'function 'db-c-debug)
               (tr (first rst) (rest rst))))))


(defun tr-first-debug (frst rst)
  (cond ((atom rst)
  (list (list 'function 'db-c-use-var)))
        (t 
    (cons (list 'function 'db-c-use-var)
           (tr (first rst) (rest rst))))))


(defun tr-list (frst rst)
  (cond ((and (atom rst) (endp frst))
   (list (list 'function 'db-c-empty-list)))
        ((atom rst)
   (list (list 'db-list (tr-sequence (tr (first frst) (rest frst))))))
    (t
         (cons (list 'db-list (tr-sequence (tr (first frst) (rest frst))))
           (tr (first rst) (rest rst))))))

(defun tr-dotted-list (frst rst)
  (cond ((atom rst)
       (list (list 'db-dotted-list (tr-dotted-sequence (tr (first frst)
                                     (if (atom (rest frst))
                                     (list (rest frst))
                                      (rest frst)))))))
    (t
         (cons (list 'db-dotted-list (tr-dotted-sequence (tr (first frst)
                                   (if (atom (rest frst))
                                     (list (rest frst))
                                    (rest frst)))))
               (tr (first rst) (rest rst))))))

(defun tr-function (frst rst)
  (cond ((atom rst)
      (list (list 'function (debug-symbol (function-s-symbol frst)))))
  (t 
      (cons (list 'function (debug-symbol (function-s-symbol frst)))
           (tr (first rst) (rest rst))))))

(defun tr-repeat (frst rst)
  (cond ((atom rst)
    (list (list 'db-repeat 
              (tr-sequence (tr (first (rest frst)) (rest (rest frst))))
                    (list 'function 'db-c-nothing))))
       (t
    (list (list 'db-repeat
        (tr-sequence (tr (first (rest frst)) (rest (rest frst))))
              (tr-sequence (tr (first rst) (rest rst))))))))

(defun tr-try (frst rst)
  (cond ((atom rst)
      (list (list 'db-try 
                (tr-sequence (tr (first (rest frst)) (rest (rest frst))))
              (list 'function 'db-c-nothing))))
    (t
         (list (list 'db-try
        (tr-sequence (tr (first (rest frst)) (rest (rest frst))))
              (tr-sequence (tr (first rst) (rest rst))))))))


(defun tr-or (frst rst)
  (cond ((endp (rest (rest frst)))
       ;; Is or of one single test , transform it to a sequence
    (cond ((endp rst)
             (list (tr-sequence (tr (first (second frst)) (rest (second frst))))))
              (t
       (list (tr-sequence (tr (first (second frst)) (append (rest (second frst)) rst)))))))
        ((atom rst)
  (list (list 'db-or 
          (cons 'list (mapcar #'(lambda (l) (tr-sequence (tr (first l) (rest l)))) (rest frst)))
              (list 'function 'db-c-nothing))))
    (t
         (list (list 'db-or
          (cons 'list (mapcar #'(lambda (l) (tr-sequence (tr (first l) (rest l)))) (rest frst)))
              (tr-sequence (tr (first rst) (rest rst))))))))


(defun tr-named-rule-1 (name syntax-exp)
  (list 'setf
      (list 'symbol-function (list 'quote (debug-symbol name)))
   (tr-sequence (tr syntax-exp nil))))

(defun tr-named-rule-2 (name syntax-exp)
  (list 'defun (debug-symbol name) (list 'e 'a 'r)
 (list 'funcall (tr-sequence (tr syntax-exp nil)) 'e 'a 'r)))

(defun tr-rule (name syntax-exp)
  (list 'setf
   (list 'symbol-function (list 'quote (debug-symbol name)))
        (list 'db-call-exp (tr-sequence (tr (first syntax-exp) (rest syntax-exp))))))

;; *****************************************************************************
;; Change a symbol in a symbol in the namespace of DEBUGGER
;; *****************************************************************************

(defun debug-symbol (symbol)
  (intern (symbol-name symbol) "DEBUGGER"))

;; ********************************************************************************************************************
;; Check if a named definition depends on another named definition 
;; ********************************************************************************************************************

(defun depends-on (def1 def2 def-exp)
  (let ((checked (make-hash-table)))
    (depends-on-1 (gethash def1 def-exp) def1 def2 def-exp checked)))

(defun depends-on-1 (def1-exp def1 def2 def-exp checked)
  (cond ((listp def1-exp)
      ;;(some #'(lambda (exp) (princ exp) (terpri) (depends-on-1 exp def1 def2 def-exp checked)) def1-exp))
      (depends-on-2 def1-exp def1 def2 def-exp checked))
        ((eq def1-exp def2) t)
        ((eq def1-exp def1)
    (if (eq def1 def2) t nil))
      ((gethash def1-exp def-exp)
  (cond ((gethash def1-exp checked)
         (if (eq def1-exp def2) t nil))
           (t
         (setf (gethash def1-exp checked) t)
     (depends-on-1 (gethash def1-exp def-exp) def1 def2 def-exp checked))))
  (t nil)))

(defun depends-on-2 (exp def1 def2 def-exp checked)
  (cond ((null exp) nil)
 ((atom exp)
     (depends-on-1 exp def1 def2 def-exp checked))
    (t
         (or (depends-on-1 (first exp) def1 def2 def-exp checked)
          (depends-on-2 (rest exp) def1 def2 def-exp checked)))))
;; *******************************************************************************************************************
;; Generates a sorted list of named definitions , so that a definition comes after the ones it dependends on
;; *******************************************************************************************************************

(defun sort-by-dependence (def-exp)
  (let ((lst nil))
    (maphash #'(lambda (x y) (push x lst)) def-exp)
    (sort-by-dependence-1 lst def-exp)))

(defun sort-by-dependence-1 (lst def-exp)
  ;; We can't use sort because depends-on is not an order relation
  (let ((depend nil)
    (depend-not nil))
    (cond ((endp lst) lst)
        ((endp (rest lst)) lst)
     (t
        (multiple-value-setq (depend depend-not) (sort-by-dependence-split (first lst) (rest lst) def-exp depend depend-not))
      (append (sort-by-dependence-1 depend def-exp) (cons (first lst) (sort-by-dependence-1 depend-not def-exp)))))))

(defun sort-by-dependence-split (def lst def-exp depend depend-not)
  (cond ((endp lst) (values depend depend-not))
  ((depends-on def (first lst) def-exp)
    (sort-by-dependence-split def (rest lst) def-exp (cons (first lst) depend) depend-not))
 (t
      (sort-by-dependence-split def (rest lst) def-exp depend (cons (first lst) depend-not)))))
  
;; ********************************************************************************************************************
;; Now we can put finally everything together and start thinking on realy parsing the syntax source file
;; and generating the debuggenerating code
;; ********************************************************************************************************************


(defstruct function-s symbol)

;; *************************************************************************************
;; Process the parsed syntax definitions , changing the references to named syntax
;; definitions to (s-function name)
;; *************************************************************************************

(defun process-parsed-def (def subdef)
  (cond ((null def) '())
        ((listp def)
   (cons (process-parsed-def (first def) subdef)
         (process-parsed-def (rest def) subdef)))
 ((gethash def subdef)
   (make-function-s :symbol def))
 (t def)))

;; *****************************************************************************************************
;; Generate code to transform expressions in debugging expressions
;; *****************************************************************************************************
;;
;; the generated code should be of the form , given a list of elements of the form ((n1 f1) .... (nm fn))
;; *******************************************************************************************************


(defun generate-add-code-exp (n-f-lst)
  (let ((case-lst (list (list 'common-lisp::function (list 'add-code-function
                                                           'exp
                                                           'free
                                                           'position
                                                           'source))
                        (list 't (list 'cond
                                       (list (list 'not
                                                   (list 'symbolp 'operator))
                                             'exp)
                                       (list (list 'macro-function 'operator)
                                             (list 'add-code-exp-macro
                                                   'exp
                                                   'free
                                                   'position
                                                   'source))
                                       (list (list 'not (list
                                                         'special-operator-p
                                                         'operator))
                                             (list 'add-code-exp-function
                                                   'exp
                                                   'free
                                                   'position
                                                   'source))
                                       (list 't 'exp))))))
    (mapc #'(lambda (n-f) (push (list (list 'quote (first n-f)) (list
                                                                 (debug-symbol
                                                                  (rest n-f))
                                                                 'exp))
                                case-lst))
          n-f-lst)
    (setf case-lst (cons 'case (cons 'operator case-lst)))
    (list 'defun
          'add-code-exp
          (list 'exp 'free 'position 'source)
          (list 'cond (list (list 'null 'exp) 'nil) (list (list 'and
                                                                (list
                                                                 'proper-list-p
                                                                 'exp)
                                                                (list 'atom
                                                                      (list
                                                                       'first
                                                                       'exp))
                                                                (list 'not
                                                                      (list
                                                                       'null
                                                                       'exp)))
                                                          (list 'let
                                                                (list (list
                                                                       'operator
                                                                       (list
                                                                        'first
                                                                        'exp)))
                                                                case-lst))))))


;; ***********************************************************************************
;; Checks if a list is a proper list (not a dotted list)
;; ***********************************************************************************

(defun proper-list-p (lst)
  (cond ((null lst) t)
     ((listp lst) (proper-list-p (cdr lst)))
    (t nil)))

;; ***********************************************************************************
;; Special case for the function special-operator
;; ***********************************************************************************

(defun add-code-function (exp free position source)
  (cond ((listp (second exp))
      (list 'function (add-code-exp (second exp) free position source)))
        (t exp)))

;; ************************************************************************************
;; Process the syntax definition file 
;; ************************************************************************************

(defun process-definition-file (definition-file &optional (destination "debugcode"))
  (declare (string definition-file))
  ;; Allow for failing of transformation , to jump out of the system
  (catch 'error
         (let ((subdef (make-hash-table))  ;; Hash table for named definitions
               (tmpsource (format nil "~A.lisp" destination))   ;; Source to generate tempcode in
               (definition nil) ;; Holds transformed definition
               (name nil) ;; name of function
               (n-f-lst nil)) ;; list of names of syntax definitions
           (declare (string tmpsource))
           ;; Read the syntax defintion file and parse it
           (setf definition (parse-definition-start (source-to-string definition-file) subdef))
           ;; Write out the code that implement the syntax definitions in lisp code
           (with-open-file
            (h tmpsource :direction :output :if-exists :supersede)
            ;; Write out some comments
            (write ";;; **********************************************************" :stream h :escape nil :readably nil) (terpri h)
            (write ";;; Generated code by Jabberwocky " :stream h :escape nil :readably nil) (terpri h)
            (write ";;; This code will transform lisp code adding instrumentation " :stream h :escape nil :readably nil) (terpri h)
            (write ";;; code for debugging " :stream h :escape nil :readably nil) (terpri h)
            (write ";;; Generated on " :stream h :escape nil :readably nil)
            (write (get-time) :stream h :escape nil :readably nil) (terpri h)
            (write ";;; **********************************************************" :stream h :escape nil :readably nil) (terpri h)
            (terpri h)
            ;;(write "(in-package \"DEBUGGER\")" :stream h :escape nil) (terpri h)
            ;; First write out the named syntax definitions 
            (mapc #'(lambda (def)
                            (cond ((depends-on def def subdef)
                                   ;; We have recursive code so we use the recursive version of transformation
                                   (write (tr-named-rule-2 def (process-parsed-def (gethash def subdef) subdef))
                                          :stream h :escape t :radix nil :base 10 :pretty t
                                          :level nil :length nil :gensym nil :array t)
                                   (terpri h))
                                  (t
                                   ;; We have non recursive code so we use the non recursive version of transformation
                                   (write (tr-named-rule-1 def (process-parsed-def (gethash def subdef) subdef))
                                          :stream h :escape t :radix nil :base 10 :pretty t
                                          :level nil :length nil :gensym nil :array t)
                                   (terpri h))))
                  (sort-by-dependence subdef))
            ;; Next we write out the unnamed syntax definitions
            (mapc #'(lambda (def)
                            (cond ((or (not (listp def)) (not (atom (first def))))
                                   (message "Definition found which is not a list" :warn t)
                                   (throw 'error nil))
                                  (t
                                   ;; Write the definition code
                                   (setf name (read-from-string (format nil "d-~A" (first def))))
                                   (push (cons (first def) name) n-f-lst)
                                   (write (tr-rule name (process-parsed-def def subdef))
                                          :stream h :escape t :radix nil :base 10 :pretty t
                                          :level nil :length nil :gensym nil :array t)
                                   (terpri h))))
                  definition)
            ;; Generate the add-code-exp fucntion based on first names in list and generated code
            (write (generate-add-code-exp n-f-lst)
                   :stream h :escape t :radix nil :base 10 :pretty nil
                   :level nil :length nil :gensym nil :array t))
           ;; Compile the generated file
           #+:CMU
           (compile-file tmpsource :output-file (format nil "cmucl/~A.x86f" destination))
           #+:SBCL
           (compile-file tmpsource :output-file (format nil "sbcl/~A.fasl" destination))
           #+CLISP
           (compile-file tmpsource :output-file (format nil "clisp/~A.fas" destination))
           #+:acl-socket
           (compile-file tmpsource :output-file (format nil "acl/~A.fasl" destination))
           #+GCL
           (compile-file tmpsource :output-file (format nil "gcl/~A.o" destination))
           ;; Load the compiled file in memory
           #+:CMU
           (load (format nil "cmucl/~A.x86f" destination))
           #+:SBCL
           (load (format nil "sbcl/~A.fasl" destination))
           #+CLISP
           (load (format nil "clisp/~A.fas" destination))
           #+:acl-socket
           (load (format nil "acl/~A.fasl" destination))
           #+GCL
           (load (format nil "gcl/~A.o" destination))
           )))

;; *************************************************************************************
;; Produces a string containing the current time
;; *************************************************************************************

(defun get-time ()
  (multiple-value-bind (hour minute second day month year)
      (get-decoded-time)
    (format nil "~A:~A:~A ~A/~A/~A" hour minute second day month year)))

;; *************************************************************************************
;; *************************************************************************************
;; Code called during instrumentation (is transforming source in debugged source)
;; *************************************************************************************
;; *************************************************************************************


;; *************************************************************************************
;; add-code to an expression 
;; *************************************************************************************

(defun add-code (exp free position source)
;;  (db-c-set-exp exp)
  (cond ((null exp) exp) ;; Case of nil expressions
       ((not (proper-list-p exp)) exp)
      ((add-code-exp exp free position source))
   (t
        (message (format nil "Unable to analyse know structure , exp : ~A" exp) :warn t)
     (when (gethash exp position)
        (highlight-error (first (gethash exp position))))
        exp)))

;; *******************************************************************************************************
;; Add debugging code to a function call
;; *******************************************************************************************************

(defun add-code-exp-function (exp free position source)
  (cons (first exp) (mapcar #'(lambda (arg)
                      (add-debug-point arg free position source))
                    (rest exp))))

;; *****************************************************************************************************
;; Add debugging code to a macro call if this is possible
;; *****************************************************************************************************

(defun add-code-exp-macro (exp free position source)
  (if **debug-macro**
      (add-code-exp (macroexpand-1 exp) free position source)
    exp))


;; *************************************************************************************
;; Do the work of adding debug code 
;; *************************************************************************************

(defun add-debug-point (exp variables position source)
  (let ((b-e (gethash exp position)))
    (cond ((and b-e (listp exp) (not (null exp))) ;; Breakpoint must have source info and must be a functioncall
           (set-possible-breakpoint source (first b-e) (1+ (rest b-e)))
           (list 'progn
                 (list 'debug-point
                       source ;;(list 'quote source)               ;; Source id
                       (first b-e)     ;; begin-pos in source , number if defined else nil
                       (if b-e (1+ (rest  b-e)) nil)     ;; end-pos in source , number if defined else nil
                       (get-lexical-binding variables))
                 ;;    (list 'ds (add-code exp variables position source))))
                 (ds (add-code exp variables position source)))) 
          (t exp))))

;; **************************************************************************************
;; Do the work of adding the first breakpoint code
;; **************************************************************************************

(defun add-debug-point-first (exp variables position source)
  (let ((b-e (gethash exp position)))
    (when b-e (set-possible-breakpoint source (first b-e) (1+ (rest b-e))))
    (list 'progn (list 'debug-point
                       source ;; (list 'quote source)
                       (first b-e)
                       (if b-e (1+ (rest b-e)) nil)
                       (get-lexical-binding variables))
          nil)))

;;; ***************************************************************************
;;; Get a assoc list of the variables binded in the lexical environment
;;; (get-lexical-binding) should eturn the code to retrieve the lexical
;;; environment for variables
;;; ***************************************************************************


#+CLISP
(defun get-lexical-binding (vars)
  '(clisp-env (EXT::THE-ENVIRONMENT)))

#+:CMU
(defun get-lexical-binding (vars)
  (let* ((result (gensym))
         (check-var (mapcar #'(lambda (v) 
                                      `(handler-case (push (cons ',v ,v) ,result)
                                                     (error ())))
                            vars)))
    `(lambda () (let ((,result nil))
                  ,@check-var
                  ,result))))

#+:acl-socket
(defun get-lexical-binding (vars)
  (let* ((result (gensym))
         (check-var (mapcar #'(lambda (v) 
                                      `(handler-case (push (cons ',v ,v) ,result)
                                                     (error ())))
                            vars)))
    `(lambda () (let ((,result nil))
                  ,@check-var
                  ,result))))

#+:SBCL
(defun get-lexical-binding (vars)
  (let* ((result (gensym))
         (check-var (mapcar #'(lambda (v) 
                                      `(handler-case (push (cons ',v ,v) ,result)
                                                     (error ())))
                            vars)))
    `(lambda () (let ((,result nil))
                  ,@check-var
                  ,result))))

#+GCL
(defun get-lexical-binding (vars)
  (let* ((result (gentemp))
         (check-var (mapcar #'(lambda (v) 
                                      `(handler-case (push (cons ',v ,v) ,result)
                                                     (error ())))
                            vars)))
    `(lambda () (let ((,result nil))
                  ,@check-var
                  ,result))))




;;; ***************************************************************************
;;; Get the lexical envrionment in CLISP
;;; ***************************************************************************

#+CLISP
(defun clisp-env (env)
  (let ((result nil))
    (do ((i 0 (1+ i))
         (l (length env)))
      ((or (>= i l) (null (aref env i))) result)
      (cond ((typep (aref env i) 'SIMPLE-VECTOR)
             (setf result (nconc result (clisp-env (aref env i)))))
            ((aref env i)
             (push (cons (aref env i) (aref env (1+ i)))
                   result)
             (setf i (1+ i)))))))

;; **************************************************************************************
;; Does the work of transforming a piece of code in debugged code , is called by 
;; debug-open-file
;; **************************************************************************************

(defun add-debugging-code (exp position source)
  (db-c-clear)
  (cond ((atom exp) exp) ;; Case of nil expression
        ((add-code-exp exp nil position source))
        (t 
         exp)))

;; ##################################################
;; ##################################################
;; ##################################################
;; ###### SECION 4 ##################################
;; ##################################################
;; ##################################################
;; ##################################################

;; *************************************************************************************
;; *************************************************************************************
;; Code , to generate java code , so that the java editor can analyze the syntax of 
;; lisp expressions
;; *************************************************************************************
;; *************************************************************************************
;;
;; (analyzer-array (s1...sn) ==> {s1,...,sn}
;; (java-identifier symbol) ==> generates a valid java-identifier
;; (to-java ()) ==> () or (new EndLispAnalyzer())
;; (to-jave exp) where exp is a atom ==> (new AfterDotAnalyzer ((to-java (list exp))))
;; (to-java (**string** | r)) ==> (new StringLispAnalyzer() | (to-jva rst))
;; (to-java (**var** | r)) ==> (new VarLispAnalyzer() | (to-java rst))
;; (to-java (**variable2** | r)) ==> (new VariableLispAnalyzer() | (to-java rst))
;; (to-java (**debug** | r)) ==> (new CallLispAnalyzer() | (to-java rst))
;; (to-java (**debug-first** | r)) ==> (new CallLispAnalyzer() | (to-java rst))
;; (to-java ((function-s symbol) r)) ==> (new (java-identifier symbol)() | (to-java rst))
;; (to-java ((a1 ... an) | r)) ==> (new BranchLispAnalyzer(new LispAnalyzer[] (analyzer-array (to-java (a1 ...an)))) | (to-java r))
;; (to-java ((**repeat** a1 ... an) | r))  ==> (new TryLispAnalyzer(new LispAnalyzer[] (analyzer-array (to-java (a1 ...an)))) | (to-java r)) (1)
;; (to-java ((**repeat** a1 ... an) | r)) ==> (new TryLispAnalyzerSimple(new LispAnalyzer[] (analyzer-array (to-java (a1 .. an)))) | (to-java r)) (2)
;; (to-java ((**try** a1 ... an) | r)) ==> (new TryOnceLispAnalyzer(new LispAnalyzer[] (analyzer-array (to-java (a1 ...an)))) | (to-java r))
;; (to-java ((**or** l1 ... ln) | r)) ==> (new ChoiceLispAnalyzer(new LispAnalyzer[][] (analyzer-array ((analyzer-array (to-java l1))
;;                                                                                                      ...
;;                                                                                                      (analyzer-array (to-java ln)))))
;;                                         (to-java r))
;; (to-java (symbol | r)) ==> (new TokenLispAnalyzer(symbol) | (to-java r))
;;
;; (1) if (to-java-simple (a1 ... an)) gives nil
;; (2) if (to-java-simple (a1 ... an)) gives T
;;
;; (to-java-simple (a1 ... an)) gives nil if there exist a ai of the form (**repeat** ...)
;;                                        or of the form (**try** ...) or of the form
;;                                        (**or** b1 ... bn) with one (to-java-simple bi) = nil
;; *****************************************************************************

;; *****************************************************************************
;; Given a list of strings , convert this to a Java literal array
;; *****************************************************************************

(defun analyzer-array (lst)
  (format nil "{~A}" (analyzer-array-1 lst)))
(defun analyzer-array-1 (lst)
  (cond ((endp lst) "")
 ((endp (rest lst))
      (first lst))
      (t
   (concatenate 'string (first lst) "," (analyzer-array-1 (rest lst))))))

;; Tests
;; (analyzer-array '()) ==> {}
;; (analyzer-array '("a")) ==> {a}
;; (analyzer-array '("a" "b")) ==> {a,b}


;; *****************************************************************************
;; Generate java code to analyze lisp expressions
;; (to-java end)
;; lst == list of syntax definitions to generate code for
;; end == indicates if we must generate the end of list test (we don't do this
;;        in a repeat,try situation or afterdot situation,
;;        but we do it in a branche situation
;; *****************************************************************************


(defun to-java (lst end)
  (cond ((null lst) (if end (list "new EndLispAnalyzer()") '()))
        ((atom lst)
         (list (format nil 
                       "new AfterDotAnalyzer (~A)"
                       (first (to-java (list lst) nil)))))
        (t 
         (let ((frst (first lst)) (rst (rest lst)))
           (cond ((eq frst **string**) (cons (to-java-string) (to-java rst end)))
                 ((eq frst **var**) (cons (to-java-var) (to-java rst end)))
                 ((eq frst **variable2**) 
                  (cons (to-java-variable) (to-java rst end)))
                 ((eq frst **debug**) (cons (to-java-call) (to-java rst end)))
                 ((eq frst **debug-first**) (cons (to-java-call) (to-java rst
                                                                          end)))
                 ((function-s-p frst) (cons (to-java-function (function-s-symbol
                                                               frst))
                                            (to-java rst end)))
                 ((listp frst) (cond ((eq (first frst) **repeat**) 
                                      (cons (to-java-repeat (rest frst))
                                            (to-java rst end)))
                                     ((eq (first frst) **try**)
                                      (cons
                                       (to-java-try (rest frst))
                                       (to-java rst end)))
                                     ((eq (first frst) **or**) 
                                      (cons (to-java-or (rest frst)) (to-java
                                                                      rst
                                                                      end)))
                                     (t (cons (to-java-branch frst)
                                              (to-java rst end)))))
                 (t (cons (to-java-symbol frst) (to-java rst end))))))))

;; *****************************************************************************
;; Generate the code for the check for a string
;; *****************************************************************************

(defun to-java-string ()
  "new StringLispAnalyzer()")
  
;; *****************************************************************************
;; Generate the code for the check for a element
;; *****************************************************************************

(defun to-java-var ()
  "new VarLispAnalyzer()")

;; ****************************************************************************
;; Generate the code for the check for a variable
;; ****************************************************************************

(defun to-java-variable ()
  "new VariableLispAnalyzer()")

;; ****************************************************************************
;; Generate the code for the check for a call of a macro or function
;; ****************************************************************************

(defun to-java-call ()
  "new CallLispAnalyzer()")

;; ***************************************************************************
;; Generate the code for applying a named syntax rule
;; ***************************************************************************

(defun to-java-function (function)
  (format nil "new ~A()" (java-identifier function)))

;; ***************************************************************************
;; Check if for lst=(a1 ... an) one of the ai is of the form :
;;   (**repeat** ...)
;;   (**try** ...)
;;   (**or** ...bi..) where (to-java-simple bi) ==> nil
;; ***************************************************************************

;;(defun to-java-simple (lst)
;;  (cond ((listp lst)
;;   (every #'(lambda (x)
;;         (cond ((listp x)
;;            (cond ((eq (first x) **repeat**) nil)
;;                           ((eq (first x) **try**) nil)
;;                   ((eq (first x) **or**)
;;                    (every #'(lambda (y)
;;                              (to-java-simple (list y)))
;;                         (rest x)))
;;                       (t (to-java-simple x))))
;;        (t t)))
;;           lst))
;;      (t t)))

(defun to-java-simple (lst)
  nil)
;; ***************************************************************************
;; Generate the code for a repeating check
;; ***************************************************************************

(defun to-java-repeat (lst)
  (if (to-java-simple lst)
      (format nil
              "new TryLispAnalyzerSimple(new LispAnalyzer[] ~A)"
              (analyzer-array (to-java lst nil)))
    (format nil
            "new TryLispAnalyzer(new LispAnalyzer[] ~A)"
            (analyzer-array (to-java lst nil)))))

;; ***************************************************************************
;; Generate the code for a try once
;; ***************************************************************************

(defun to-java-try (lst)
  (format nil
   "new TryOnceLispAnalyzer(new LispAnalyzer[] ~A)"
       (analyzer-array (to-java lst nil))))

;; **************************************************************************
;; Generate the code for a choice
;; **************************************************************************

(defun to-java-or (or-lst)
  (if (endp (rest or-lst))
      (to-java-sequence (first or-lst) nil)
    (format nil
            "new ChoiceLispAnalyzer(new LispAnalyzer[][] ~A)"
            (analyzer-array (mapcar #'(lambda (lst)
                                              (format nil
                                                      "new LispAnalyzer[] ~A"
                                                      (analyzer-array (to-java
                                                                       lst
                                                                       nil))))
                                    or-lst)))))

;; *****************************************************************************
;; Analyze the branching of a lisp expression
;; *****************************************************************************

(defun to-java-branch (branch)
  (format nil
     "new BranchLispAnalyzer(new LispAnalyzer[] ~A)"
          (analyzer-array (to-java branch t))))

;; ****************************************************************************
;; Analyze a symbol 
;; ****************************************************************************

(defun to-java-symbol (symbol)
  (if (keywordp symbol)
      (format nil "new TokenLispAnalyzer(\":~A\")" (string-upcase symbol))
    (format nil "new TokenLispAnalyzer(\"~A\")" (string-upcase symbol))))


;; *****************************************************************************
;; Tests of to-java
;;
;; (to-java '() t) ==> ("new EndLispAnalyzer()")
;; (to-java '(a) t) ==> ("new TokenLispAnalyzer(\"A\")" "new EndLispAnalyzer()")
;; (to-java '(a b) t) ==> ("new TokenLispAnalyzer(\"A\")" "new TokenLispAnalyzer(\"B\")" "new EndLispAnalyzer()")
;; (to-java (list **string** 'a) t) ===> ("new StringLispAnalyzer()" "new TokenLispAnalyzer(\"A\")" "new EndLispAnalyzer()")
;; (to-java (list **var**) nil) ==> ("new VarLispAnalyzer()")
;; (to-java (list **variable2**) nil) ==> ("new VariableLispAnalyzer()")
;; (to-java (list **debug**) nil) ==> ("new CallLispAnalyzer()")
;; (to-java (list **debug-first**) nil) ==> ("new CallLispAnalyzer()")
;; (to-java (list (list **var** 'a) 'b) nil) ==> ("new BranchLispAnalyzer(new LispAnalyzer[] {new VarLispAnalyzer(),new TokenLispAnalyzer(\"A\"),new EndLispAnalyzer()})" "new TokenLispAnalyzer(\"B\")")
;; (to-java (list (list **repeat** 'a 'b)) nil) ==> ("new TryLispAnalyzer(new LispAnalyzer[] {new TokenLispAnalyzer(\"A\"),new TokenLispAnalyzer(\"B\")})")
;; (to-java (list (list function-s 'lambda-list)) nil) ==>("new LAMBDA_LIST()")
;; (to-java (list (list **try** 'a 'b)) nil) ==> ("new TryOnceLispAnalyzer(new LispAnalyzer[] {new TokenLispAnalyzer(\"A\"),new TokenLispAnalyzer(\"B\")})")
;; (to-java (list (list **or** (list 'a 'b) (list **var**))) nil) ==>("new ChoiceLispAnalyzer(new LispAnalyzer[][] {new LispAnalyzer[] {new TokenLispAnalyzer(\"A\"),new TokenLispAnalyzer(\"B\")},new LispAnalyzer[] {new VarLispAnalyzer()}})")
;; 
;; *****************************************************************************

;; *****************************************************************************
;; Generate the code for a sequence of analyzers
;; *****************************************************************************

(defun to-java-sequence (lst end)
  (format nil
          "new SequenceLispAnalyzer(new LispAnalyzer[] ~A)"
     (analyzer-array (to-java lst end))))
;; Test
;; (to-java-sequence '(a b) t) ==> "new SequenceLispAnalyzer(new LispAnalyzer[] {new TokenLispAnalyzer(\"A\"),new TokenLispAnalyzer(\"B\"),new EndLispAnalyzer()})"
;;

;; ****************************************************************************************************
;; Generate the code for a named syntax rule
;; ****************************************************************************************************
(defun named-to-java (symbol lst)
  (let ((name (java-identifier symbol)))
    (declare (string name))
    (concatenate
     'string
     "class " name " extends LispAnalyzer" (string #\newline)
     " {" (string #\newline)
     "  private static LispAnalyzer fun=" (to-java-sequence (list lst) nil) ";" (string #\newline)
     "  public " name "()" (string #\newline)
     "   {" (string #\newline)
     "    super();" (string #\newline)
     "   }" (string #\newline)
     "  public boolean analyze(LispListElement _list,CodeElementIterator _listelements,AnalyzerList todo)" (string #\newline)
     "   {" (string #\newline)
     "    return fun.analyze(_list,_listelements,todo);" (string #\newline)
     "   }" (string #\newline)
     "  public boolean findHints(CodeElement _element,LispListElement _list,CodeElementIterator _listelements,Vector _hints,AnalyzerList todo,CodeDocument doc)" (string #\newline)
     "   {" (string #\newline)
     "    return fun.findHints(_element,_list,_listelements,_hints,todo,doc);" (string #\newline)
     "   }" (string #\newline)
     " }" (string #\newline))))

;; **************************************************************************************
;; Process the syntax definition file, producing java code , which will be used by the
;; editor to do syntax analysis
;; **************************************************************************************

(defun process-definition-file-to-java (definition-file)
  (declare (simple-string definition-file))
  ;; Allow for failing of generating java code (errors in syntax file) to jump out of the system
  (catch 'error
         (let ((subdef (make-hash-table)) 
               ;; Hash table for named definitions
               (definition nil)
               ;; Holds transformed definitions
               (f-n-lst nil))
           ;; List of (f . n) f=first element of syntax definition , n is name
           ;; definition
           ;; Make sure that we don't generate duplicate java names
           (init-java-identifier)
           ;; Read the syntax definition file and parse it
           (setf definition (parse-definition-start (source-to-string
                                                     definition-file)
                                                    subdef))
           ;; Generate the java code to analyze lisp expressions
           (with-open-file (h "java/IDE/LispSyntax/LispAnalyzerGen.java"
                              :direction
                              :output
                              :if-exists
                              :supersede)
                           ;; Write out some classical humbug of java
                           (write-line "/*" h)
                           (write-line
                            " * Code generated by process-definition-file-to-java in LISP"
                            h)
                           (write-line " */" h)
                           (write-line "package IDE.LispSyntax;" h)
                           (write-line "import IDE.CodeEditor.*;" h)
                           (write-line "import IDE.CodeEditor.Parser.*;" h)
                           (write-line "import java.util.*;" h)
                           (write-line "/*" h)
                           (write-line
                            " * The class definitions for the named rules in the syntaxdefinition"
                            h)
                           (write-line " */" h)
                           ;; First write out the named syntax
                           (maphash
                            #'(lambda (key def)
                                      (write-line "/*" h)
                                      (write-line (format nil
                                                          " * ~A"
                                                          (java-comment-syntax
                                                           def))
                                                  h)
                                      (write-line " */" h)
                                      (write-line
                                       (named-to-java key (process-parsed-def
                                                           def
                                                           subdef))
                                       h))
                            subdef)
                           ;; Next write out the unamed syntax definitions
                           (write-line "public class LispAnalyzerGen" h)
                           (write-line "{" h)
                           ;; Create the java code for the unnamed rules
                           (mapc
                            #'(lambda (def)
                                      (cond ((or (not (listp def)) (not (atom
                                                                         (first
                                                                          def)))) 
                                             (message
                                              "Definition found which is not a list"
                                              :warn
                                              t)
                                             (throw 'error nil))
                                            (t 
                                             ;; Save the symbol and name of the unnamed syntax rules
                                             (push
                                              (cons (first def) (java-identifier
                                                                 (first def)))
                                              f-n-lst)
                                             ;; Genearate the java code to do the analyze
                                             (write-line "/*" h)
                                             (write-line
                                              (format nil
                                                      " * ~A"
                                                      (java-comment-syntax def))
                                              h)
                                             (write-line " */" h)
                                             (write-line
                                              (format nil
                                                      " private static LispAnalyzer ~A=~A;"
                                                      (java-identifier (first
                                                                        def))
                                                      (to-java-sequence
                                                       (process-parsed-def def
                                                                           subdef)
                                                       t))
                                              h))))
                            definition)
                           ;; Next create the method who does the analyze
                           (write-line "/*" h)
                           (write-line
                            " * Method to do the analyze of the lisp expressions"
                            h)
                           (write-line " */" h)
                           (write-line
                            "  public static boolean analyze(LispListElement list)"
                            h)
                           (write-line "  {" h)
                           (write-line
                            "     CodeElementIterator iter=list.getElements().makeIterator();"
                            h) ;
                           (write-line
                            "     AnalyzerList todo=new AnalyzerList();"
                            h)
                           (write-line "     if (list.isEmpty())" h)
                           (write-line "        {" h)
                           (write-line "         return false;" h)
                           (write-line "        }" h)
                           (mapc
                            #'(lambda (f-n)
                                      (write-line
                                       (format nil
                                               "     else if (list.startsWith(\"~A\"))"
                                               (string-upcase (first f-n)))
                                       h)
                                      (write-line "     {" h)
                                      (write-line
                                       (format nil
                                               "               return LispAnalyzerGen.~A.analyze(list,iter,todo);"
                                               (rest f-n))
                                       h)
                                      (write-line "     }" h))
                            f-n-lst)
                           (write-line "     else" h)
                           (write-line "     {" h)
                           (write-line "      return false;" h)
                           (write-line "     }" h)
                           (write-line "  }" h)
                           ;; Next create the method who finds the hints
                           (write-line "/*" h)
                           (write-line
                            " * Method to find possible hints"
                            h)
                           (write-line " */" h)
                           (write-line
                            "  public static boolean findHints(CodeElement element,Vector hints,LispListElement list,CodeDocument doc)"
                            h)
                           (write-line "  {" h)
                           (write-line
                            "     CodeElementIterator iter=list.getElements().makeIterator();"
                            h) ;
                           (write-line
                            "     AnalyzerList todo=new AnalyzerList();"
                            h)
                           (write-line "     if (list.isEmpty())" h)
                           (write-line "        {" h)
                           (write-line "         return false;" h)
                           (write-line "        }" h)
                           (mapc
                            #'(lambda (f-n)
                                      (write-line
                                       (format nil
                                               "     else if (list.startsWith(\"~A\"))"
                                               (string-upcase (first f-n)))
                                       h)
                                      (write-line "     {" h)
                                      (write-line
                                       (format nil
                                               "               return LispAnalyzerGen.~A.findHints(element,list,iter,hints,todo,doc);"
                                               (rest f-n))
                                       h)
                                      (write-line "     }" h))
                            f-n-lst)
                           (write-line "     else" h)
                           (write-line "     {" h)
                           (write-line "      return false;" h)
                           (write-line "     }" h)
                           (write-line "  }" h)
                           ;; Create the java code to create a array of all the lisp keywords
                           (write-line "/*" h)
                           (write-line
                            " * Method to get a array of Lisp keywords"
                            h)
                           (write-line " */" h)
                           (write-line
                            "    public static String[] getLispKeywords()"
                            h)
                           (write-line "    {" h)
                           (write-line "     return LispKeywords;" h)
                           (write-line "    }" h)
                           (write-line "" h)
                           (write-line "/*" h)
                           (write-line
                            " * Array containing the name of all Lisp macros and functions defined in the Lisp Syntax"
                            h)
                           (write-line "*/" h)
                           (write-line
                            "    private static String[] LispKeywords = {"
                            h)
                           (let ((frst t) (lst nil))
                             ;; Find the known symbols (not just the one where a syntax is defined for)
                             (do-symbols (smb
                                          (find-package ++common-lisp-user++))
                                         (when (fboundp smb) (push
                                                              (cons smb nil)
                                                              lst)))
                             (mapc
                              #'(lambda (f-n)
                                        (if frst
                                            (write-line
                                             (format nil
                                                     "                              \"~A\""
                                                     (string-upcase (first f-n)))
                                             h)
                                          (write-line
                                           (format nil
                                                   "                              ,\"~A\""
                                                   (string-upcase (first f-n)))
                                           h))
                                        (setf frst nil))
                              (union lst f-n-lst)))
                           (write-line
                            "                                           };"
                            h)
                           ;; Close the definition of the class
                           (write-line "}" h)))))

;; **************************************************************************************
;; Write the syntax expression as comment to java
;; **************************************************************************************

(defun java-comment-syntax (lst)
  (cond ((atom lst)
     lst)
     ((endp lst) '())
   (t
        (let ((frst (first lst))
           (rst (rest lst)))
     (cons (cond ((eq frst **string**) "**string**")
                  ((eq frst **var**) "**var**")
                    ((eq frst **variable2**) "**variable2**")
              ((eq frst **debug**) "**debug**")
                ((eq frst **debug-first**) "**debug-first**")
          ((eq frst **repeat**) "**repeat**")
                ((eq frst **try**) "**try**")
                  ((eq frst **or**) "**or**")
                ((listp frst)
           (java-comment-syntax frst))
            (t frst))
                (java-comment-syntax rst))))))


;; ***************************************************************************************
;; Convert a name to a legal java identifier
;; ***************************************************************************************

(let ((used (make-hash-table))
      (transform (make-hash-table)))
  ;; Init the hash tables used here
  (defun init-java-identifier ()
    (setf transform (make-hash-table :test #'eq))
    (setf used (make-hash-table :test #'equal)))
  ;; Transform a symbol to a valid java identifier name (given the symbol we can retrieve this name)
  (defun java-identifier (symbol)
    (cond ((gethash symbol transform))
          (t
           (let ((name (princ-to-string symbol)))
             (do ((i 0 (incf i))
                  (l (length name)))
               ((>= i l) name)
               (when (not (alphanumericp (char name i)))
                 (setf (char name i) #\_)))
             (loop
              (cond ((gethash name used)
                     (setf name (concatenate 'string name "U")))
                    (t
                     (setf (gethash name used) t)
                     (setf name (string-upcase name))
                     (setf (gethash symbol transform) name)
                     (return name)))))))))

        

              
;; ##################################################
;; ##################################################
;; ##################################################
;; ###### SECTION 5 #################################
;; ##################################################
;; ##################################################
;; ##################################################

;; **************************************************************************************
;; **************************************************************************************
;; Code , used during the debugging operation itself
;; **************************************************************************************
;; **************************************************************************************
;;
;; To understand the working : remember that a call in a function is
;; replaced by (progn (debugpoint ....) (ds call))
;; So (debugpoint ...) gets called for execution of the call
;; The macro (ds call) will call a function after execution of the call



;; *****************************************************************
;; Macro , given a list of variables , generate a list of variables
;; which are binded
;; *****************************************************************

(defmacro get-binding (&rest var-lst)
  (let ((res nil))
    (mapcar #'(lambda (var)
           (push (list 'list (list 'quote var) var) res)
           (handler-case (eval var) (error () (pop res))))
       var-lst)
    (cons 'list res)))
  
;; *****************************************************************
;; Set the **display-result-call** to do after result debugging
;; *****************************************************************

(defun stop-after ()
  (setf **display-result-call** t))

;; *****************************************************************
;; Change the result of a call to the form
;; *****************************************************************

(defun change-result (expstr)
  (let ((newval (eval-expression-string expstr)))
    (when (not (eq newval **undefined**))
      (setf **values** newval)
      (refresh-watches)
      newval)))
;; *****************************************************************
;; Display the result of the previous function call 
;; *****************************************************************

(defun display-result-call (val env source begin end)
  (cond ((and (not **load**) (or **display-result-call** **run-error**))
         ;; We have a second break after the execution of the call
         ;; Define the current environment so that functions called
         ;; from the interface van reach it
         (setf **run-error** nil)
         (setf **end-debug-eventloop** nil)
         (setf **current-env** env)
         (setf **current-begin** begin)
         (setf **current-end** end)
         (setf **current-source** source)
         (setf **values** val)
         (setf **display-result-call** nil)
         ;; Give control to user interface
         (give-control-to-interface-after source begin end)
         ;; Display watchpoints if stopped in interface
         ;; Loop until from the interface a command to proceed is send (via process-incoming)
         (loop
          (cond (**end-debug-eventloop**
                 (setf **current-env** nil)
                 (setf **current-begin** 0)
                 (setf **current-end** 0)
                 (setf **current-source** "")
                 (return **values**))
                (t
                 (process-incoming nil :block t)))))   ;; Process commands from the debugger
        (t
         (setf **current-env** nil)
         (setf **current-begin** 0)
         (setf **current-end** 0)
         (setf **current-source** "")
         (setf **display-result-call** nil)
         val)))

;;******************************************************************
;; Activate debugger (after error in executing debugged code)
;; *****************************************************************

(defun activate-debugger (condition)
  (setf **run-error** t)
  (message (format nil "~A" condition) :warn t))

  
;; *****************************************************************
;; Macro to save the result of a function call
;; *****************************************************************

;;(defmacro ds (vls)
;;  (let ((val (gensym))
;; (env (gensym))
;;       (source (gensym))
;;        (begin (gensym))
;;       (end (gensym)))
;;    (if **check-error**
;;  `(let* ((,env **current-env**)
;;               (,source **current-source**)
;;             (,begin **current-begin**)
;;               (,end **current-end**)
;;         (,val (multiple-value-list (handler-case ,vls
;;                          (error (condition)
;;                             (activate-debugger condition)
;;                                 nil)))))
;;       (apply #'values (display-result-call ,val ,env ,source ,begin ,end)))
;;     `(let* ((,env **current-env**)
;;              (,source **current-source**)
;;             (,begin **current-begin**)
;;         (,end **current-end**)
;;         (,val (multiple-value-list ,vls)))
;;       (apply #'values (display-result-call ,val ,env ,source ,begin ,end))))))           

;; ******************************************************************
;; Generate the code to sace the result of a function call
;; Do not use a macro anymore to avoid overhead on the Lisp 
;; compiler
;; ******************************************************************
#-GCL
(defun ds (vls)
  (let ((val (gensym))
        (env (gensym))
        (source (gensym))
        (begin (gensym))
        (end (gensym)))
    (cond (**check-error**
           (list 'let*
                 (list
                  (list env '**current-env**)
                  (list source '**current-source**)
                  (list begin '**current-begin**)
                  (list end '**current-end**)
                  (list val (list 'multiple-value-list 
                                  (list 'handler-case 
                                        vls
                                        '(error (condition) 
                                                (activate-debugger condition)
                                                nil)))))
                 (list 'apply '#'values (list 'display-result-call 
                                              val
                                              env
                                              source
                                              begin
                                              end))))
          (**debug-after**
           (list 'let*
                 (list
                  (list env '**current-env**)
                  (list source '**current-source**)
                  (list begin '**current-begin**)
                  (list end '**current-end**)
                  (list val (list 'multiple-value-list vls)))
                 (list 'apply '#'values (list 'display-result-call
                                              val
                                              env
                                              source
                                              begin
                                              end))))
          (t
           vls))))

#+GCL
(defun ds (vls)
  (let ((val (gentemp))
        (env (gentemp))
        (source (gentemp))
        (begin (gentemp))
        (end (gentemp)))
    (cond (**check-error**
           (list 'let*
                 (list
                  (list env '**current-env**)
                  (list source '**current-source**)
                  (list begin '**current-begin**)
                  (list end '**current-end**)
                  (list val (list 'multiple-value-list 
                                  (list 'handler-case 
                                        vls
                                        '(error (condition) 
                                                (activate-debugger condition)
                                                nil)))))
                 (list 'apply '#'values (list 'display-result-call 
                                              val
                                              env
                                              source
                                              begin
                                              end))))
          (**debug-after**
           (list 'let*
                 (list
                  (list env '**current-env**)
                  (list source '**current-source**)
                  (list begin '**current-begin**)
                  (list end '**current-end**)
                  (list val (list 'multiple-value-list vls)))
                 (list 'apply '#'values (list 'display-result-call
                                              val
                                              env
                                              source
                                              begin
                                              end))))
          (t
           vls))))

;; ******************************************************************
;; End the eventloop in a debug point , allowing further processing *
;; ******************************************************************

(defun end-debug-eventloop ()
  (setf **alternate-time** -1)
  (setf **end-debug-eventloop** t))


;; *********************************************************************
;; Lisp function called in a debug point                               *
;; *********************************************************************

(defun debug-point (lisp-source begin end env)
  (when **load** (return-from debug-point))
  ;; Register the call if profiling is turned on
  (when **profile** 
    (register-call lisp-source begin end))
  ;; Define the current environment , so that functions called from
  ;; the interface can reach it
  (setf **values** nil)
  ;; (setf **end-debug-eventloop** nil)
  (setf **current-env** env)
  (setf **current-begin** begin)
  (setf **current-end** end)
  (setf **current-source** lisp-source)
  ;; Save information to do timetraveling
  (save-time env begin end lisp-source)
  ;; Give control to the user interface
  (when (must-stop-after-call lisp-source begin end)
    (setf **end-debug-eventloop** (give-control-to-interface lisp-source begin end))
    ;; Display watchpoints if stopped in interface
    ;; Loop until from the interface a command to proceed is send (via process-incoming)
    (loop
     (cond (**end-debug-eventloop**
            (return))
           (t
            (process-incoming nil :block t))))) ;; Process commands from debugger (block on read)
  (setf **current-env** nil)) ;; Ensure that we have no current environment outsidedebug point 


;; **************************************************************************************
;; Evaluate a expression string in the execution context returns the result as a value list
;; **************************************************************************************

(defun eval-expression-string (expstr)
    (handler-case (multiple-value-list 
                   (eval (list 'let
                               (mapcar #'(lambda (var-value)
                                                 (list (first var-value)
                                                       (list 'quote (rest var-value))))
                                       #+:CMU
                                       (if **current-env**
                                         (funcall **current-env**)
                                         nil)
                                       #+:acl-socket
                                       (if **current-env**
                                           (funcall **current-env**)
                                         nil)
                                       #+:SBCL
                                       (if **current-env**
                                           (funcall **current-env**))
                                       #+GCL
                                       (if **current-env**
                                           (funcall **current-env**))
                                       #+CLISP
                                       **current-env**)
                               (read-from-string expstr))))
                  (error () (return-from eval-expression-string **undefined**))))


;; ************************************************************************************
;; Evaluates a expression string returning a list (error output)
;; where: error is the error message if a error is encountered during evaluation NIL otherwise
;;        output is the result of the evaluation (nil if error is encountered)
;; ************************************************************************************

(defun eval-element (expstr)
  (let ((*error-output* (make-string-output-stream)))
    (handler-case (list NIL
                        (eval (read-from-string expstr)))
                  (error (e)
                         (return-from eval-element
                                      (list (format nil "~A" e) NIL))))))

;; ************************************************************************************
;; Returns a string representin the result of evaluation of a string or nil if a error
;; is encountered (if values are returned only the first value is shown)
;; Also suppress every error or warning
;; ************************************************************************************

(defun simple-eval-expression (expstr)
  (let ((*error-output* (make-string-output-stream)))
    (let ((result (eval-expression-string expstr)))
      (if (eq result **undefined**)
          nil
        (princ-to-string (first result))))))

;; ************************************************************************************
;; Evaluates a expression returning a list containing the values returned in formatted
;; so that java can display it in a structured way
;; ************************************************************************************

(defun eval-expression (expstr)
  (let ((*error-output* (make-string-output-stream)))
    (let ((result (eval-expression-string expstr)))
      (if (eq result **undefined**)
          nil
        (mapcar #'lisp-type-to-java result)))))

;; ************************************************************************************
;; Check if a string expression evaluates to non nil, used to implement a conditional
;; breakpoint (returns T or nil)
;; ************************************************************************************

(defun expression-evaluate-to-true (expstr)
  (let ((result (eval-expression-string expstr)))
    (if (first result)
        t
      nil)))


;; *************************************************************************************
;; Returns a list of the form (type-indicator value(s))
;; Where type-indicator is a number telling java what type the value(s) represent 
;; *************************************************************************************

(defun lisp-type-to-java (value)
  (typecase value
    (string
     (list 1 value))
    (bit-vector
     (list 2 (lisp-type-to-java-sequence value)))
    (vector
     (list 3 (lisp-type-to-java-sequence value)))
    (array ;; Array
     (list 4 (lisp-type-to-java-sequence value)))
    (character
     (list 5 value))
    (complex
     (list 6 (list (lisp-type-to-java (realpart value))
                (lisp-type-to-java (imagpart value)))))
    (float
     (list 7 value))
    #-GCL
    (function 
     (list 8 (princ-to-string value)))
    (hash-table
     (list 9 (lisp-type-to-java-hashtable value)))
    (integer
     (list 10 value))
    (ratio
     (list 11 (list (lisp-type-to-java (numerator value))
             (lisp-type-to-java (denominator value)))))
    (number
     (list 12 value))
    (package
     (list 13 (princ-to-string value)))
    (pathname
     (list 14 (princ-to-string value)))
    (random-state
     (list 15 (princ-to-string value)))
    (readtable
     (list 16 (princ-to-string value)))
    (null
     (list 17 value))
    (cons
     (cond ((is-list value)
        (list 18 (lisp-type-to-java-list value)))
        (t
          (list 19 (list (lisp-type-to-java (first value))
                        (lisp-type-to-java (rest value)))))))
    (sequence
     (list 20 (lisp-type-to-java-sequence value)))
    (stream 
     (list 21 (princ-to-string value)))
    (symbol
     (list 22 (princ-to-string value)))
    (t
     (cond ((eq value t)
      (list 23 value))
       ((typep (class-of value) 'structure-class)
         (list 24 (lisp-type-to-java-class value)))
        ((typep (class-of value) 'standard-class)
           (list 25 (lisp-type-to-java-class value)))
          (t
            (list 26 (princ-to-string value))))))) ;; Unknown type


(defun lisp-type-to-java-sequence (value)
  (mapcar #'lisp-type-to-java (coerce value 'list)))
(defun lisp-type-to-java-hashtable (value)
  (let ((result nil))
    (maphash #'(lambda (key value)
              (push (list (lisp-type-to-java value) (lisp-type-to-java value)) result))
       value)
    result))
(defun lisp-type-to-java-list (value)
  (mapcar #'lisp-type-to-java value))
;; CLISP does not work with PCL
#+:CMU
(defun lisp-type-to-java-class (value)
  (mapcar #'(lambda (slot)
        (if (slot-boundp value (PCL:slot-definition-name slot))
       (list (princ-to-string (PCL:slot-definition-name slot))
                  (lisp-type-to-java (slot-value value (PCL:slot-definition-name slot))))
       (list (princ-to-string (PCL:slot-definition-name slot))
         (lisp-type-to-java "** Unbound Slot **"))))
        (PCL:compute-slots (PCL:class-of value))))
#+CLISP
(defun lisp-type-to-java-class (value)
  (list (list "Not Implemented in CLISP" (lisp-type-to-java (with-output-to-string (strm) (describe value strm))))))

#+:acl-socket
(defun lisp-type-to-java-class (value)
  (mapcar #'(lambda (slot)
        (if (slot-boundp value (mop:slot-definition-name slot))
       (list (princ-to-string (mop:slot-definition-name slot))
                  (lisp-type-to-java (slot-value value (mop:slot-definition-name slot))))
       (list (princ-to-string (mop:slot-definition-name slot))
         (lisp-type-to-java "** Unbound Slot **"))))
        (mop:compute-slots (class-of value))))
#+:SBCL
(defun lisp-type-to-java-class (value)
  (mapcar #'(lambda (slot)
        (if (slot-boundp value (SB-PCL::slot-definition-name slot))
       (list (princ-to-string (SB-PCL::slot-definition-name slot))
                  (lisp-type-to-java (slot-value value (SB-PCL::slot-definition-name slot))))
       (list (princ-to-string (SB-PCL::slot-definition-name slot))
         (lisp-type-to-java "** Unbound Slot **"))))
        (SB-PCL:compute-slots (SB-PCL::class-of value))))

;;**************************************************************************************
;; Check if argument is a list
;; *************************************************************************************
(defun is-list (value)
  (cond ((null value) t)
   ((consp value) (is-list (cdr value)))
    (t nil)))
  
;; *************************************************************************************
;; Keep track of environments in time
;; *************************************************************************************

;; ************************************************************************************
;; Create array to store different environments to do timetraveling
;; ************************************************************************************

(defun prepare-time (length)
  (declare (fixnum length))
  (setf **time** (make-array length :initial-element nil))
  (setf **now** 0)
  (setf **end-of-time** (1- length)))

;; *************************************************************************************
;; Store current environment in time
;; *************************************************************************************

(defun save-time (env begin end source)
  (setf (aref **time** **now**) (list env begin end source))
  (setf **now** (next-time **now**)))


;; *************************************************************************************
;; Go forwards in time , negative time means we want to jump to just after the current time
;; *************************************************************************************

(defun next-time (time)
  (declare (fixnum time))
  (cond ((< time 0) (prev-time **now**))
   ((= time **end-of-time**)
         0)
   (t (1+ time))))

;; *************************************************************************************
;; Go backwards in time , negative time means we want to jump to the current time
;; *************************************************************************************

(defun prev-time (time)
  (declare (fixnum time))
  (cond ((< time 0) (prev-time (prev-time **now**)))
     ((zerop time)
       **end-of-time**)
   (t (- time 1))))

;; *************************************************************************************
;; Step back in time
;; *************************************************************************************

(defun step-back-in-time ()
  (setf **alternate-time** (prev-time **alternate-time**))
  (let ((time **alternate-time**))
    (declare (fixnum time))
    (loop
      (when (not (null (aref **time** **alternate-time**)))
        (return))
      (setf **alternate-time** (prev-time **alternate-time**))
      (when (= time **alternate-time**)
      (message "No history kept" :warn t)
 (return-from step-back-in-time)))
    (when (= **now** **alternate-time**)
      (message "You are at the current breakpoint") :warn t)
    (setf **current-env** (first (aref **time** **alternate-time**)))
    (apply #'display-time-env (rest (aref **time** **alternate-time**)))))


;; ************************************************************************************
;; Step forward in time
;; ************************************************************************************

(defun step-forward-in-time ()
  (setf **alternate-time** (next-time **alternate-time**))
  (let ((time **alternate-time**))
    (declare (fixnum time))
    (loop
      (when (not (null (aref **time** **alternate-time**)))
     (return))
      (setf **alternate-time** (next-time **alternate-time**))
      (when (= time **alternate-time**)
   (message "No History kept" :warn t)
      (return-from step-forward-in-time)))
    (when (= **now** **alternate-time**)
      (message "You are at the current breakpoint") :warn t)
    (setf **current-env** (first (aref **time** **alternate-time**)))
    (apply #'display-time-env (rest (aref **time** **alternate-time**)))))

;; *********************************************************************************
;; Load a LISP source file in memory, if there is a error returns the position in
;; the file where the error is, otherwise returns 0
;; *********************************************************************************

(defun checked-load-file (filename &key (verbose t) (print t))
  (let ((position 0)
        (string-stream (make-string-output-stream))
        (file-stream (open filename :direction :input)))
    (unwind-protect
     (with-open-stream (echo-stream (make-echo-stream file-stream string-stream))
                       (handler-case
                        (load echo-stream :verbose verbose :print print)
                        (error (err)
                               (message (format nil "Load error : ~A" err) :warn t)
                               (setf position (length (get-output-stream-string string-stream)))))
                       (close file-stream)))
    position))

;; *********************************************************************************
;; Compile a LISP source file and load the compiled file in memory, if there is a
;; a error returns nil otherwise returns T
;; *********************************************************************************

(defun checked-compile-file (filename out)
		(handler-case
	  (progn
				(compile-file filename :output-file out :verbose t :print t)
				(load out))
			(error (err) (message (format nil "Compile error : ~A" err) :warn t)))
		(terpri)) ;; Force a print of a newline 
 

;; ****************************************************************************
;; Get a string of the features defined for this session
;; ****************************************************************************

(defun get-features()
  (mapcar #'(lambda (symbol) (format NIL "~S" symbol)) *features*))










































































































