;; ************************************************************************
;; Interfact to call functions in the java environment which provides
;; the GUID
;; ************************************************************************

(in-package "DEBUGGER")

;; ************************************************************************
;; Class to represent a connection from LISP to JAVA
;; ************************************************************************

(defclass java-connection ()
  ;; Stream to send calls to java
  ((call-in :initarg :call-in :reader java-connection-call-in)
   ;; Stream to read result of calls
   (call-out :initarg :call-out :reader java-connection-call-out)
   ;; Stream to get calls from java
   (called-in :initarg :called-in :reader java-connection-called-in)
   ;; Stream to send result back to java
   (called-out :initarg :called-out :reader java-connection-called-out)
   ;; function to be called by the callback handler, by saving this we can manual call
   ;; the processing of commands send to lisp
   (handler :initarg :handler :accessor java-connection-handler)
   ;; Callback handler
   (callback :initarg :callback :accessor java-connection-callback)
   ;; Hash table to store LISP objects not known by JAVA
   (lispobjects :initarg :lispobjects :accessor java-connection-lispobjects)
   ;; Nr of entries stored in the hash table
   (nrlispobjects :initarg :nrlispobjects :accessor java-connection-nrlispobjects)
   ;; The java object referring to this connection
   (java :initarg :java :reader java-connection-java)))


;; *************************************************************************
;; Class to hold a Java Object
;; id is a fixnum, maintained in the java process which has a mapping of this
;; number to the real java object. (Pass by reference)
;; *************************************************************************

(defclass java-object ()
  ((id :initarg :id :reader java-object-id))) ;; id=0 always refers to the java object
                                              ;; representing the 'this' object
;; *************************************************************************
;; Get the id of the java object to serialize this object
;; *************************************************************************
(defun serialize-java-object (object)
  (java-object-id object))

;; *************************************************************************
;; Desrialize a srialized java object
;; (deserialize-java-object (serialize-java-object)) should represent the
;; same java object (of course not the same lisp object
;; *************************************************************************

(defun deserialize-java-object (serialized)
  (declare (fixnum serialized))
  (make-instance 'java-object :id serialized))

;; ************************************************************************
;; This code is LISP version specific 
;; Create a connection object to a java process on socketnr1 (call) and
;; socketnr2 (called) , process-incoming is a function to process called
;; commands
;; ************************************************************************

;; CMUCL version
#+:CMU
(defun make-java-connection (socketnr1 socketnr2)
  "(make-java-connecton socknr1 socknr2)
   returns a java-connection object
   socknr1 = socket where java process listen for commands
   socknr2 = socket where java sends commands for lisp"
  (let ((call-fd 0) (called-fd 0) (callback nil) (handler nil) (connection nil))
    (sleep 4)
    ;; Establish a socket connection on socketnr1 (10 tries before giving up)
    (setf call-fd (extensions::connect-to-inet-socket "127.0.0.1" socketnr1))
    ;; Give java time to accept the connection
    (sleep 4)
    ;; Establish a socket connection on socketnr1 (10 tries before giving up)
    (setf called-fd (extensions::connect-to-inet-socket "127.0.0.1" socketnr2))
    ;; Create now the java connection
    (setf connection (make-instance 'java-connection
                                    :call-in
                                    (extensions::make-fd-stream call-fd
                                                                :input
                                                                t
                                                                :buffering
                                                                :none)
                                    :call-out
                                    (extensions::make-fd-stream call-fd
                                                                :output
                                                                t
                                                                :buffering
                                                                :none)
                                    :called-in
                                    (extensions::make-fd-stream called-fd
                                                                :input
                                                                T
                                                                :buffering
                                                                :none)
                                    :called-out
                                    (extensions::make-fd-stream called-fd
                                                                :output
                                                                t
                                                                :buffering
                                                                :none)
                                    :callback
                                    nil
                                    :lispobjects
                                    (make-hash-table :test #'eql :size 100)
                                    :nrlispobjects
                                    0
                                    :java
                                    (make-instance 'java-object :id 0)))
    ;; Generate the handler for the input , must contain connection object
    (setf handler (generate-process-incoming connection))
    (setf callback (extensions::add-fd-handler called-fd :input handler))
    (setf (java-connection-handler connection) handler)
    (setf (java-connection-callback connection) callback)
    ;; Update connection
    (funcall handler)
    ;; return connection
    connection))

;; CLISP version
#+CLISP
(defun make-java-connection (socketnr1 socketnr2)
  "(make-java-connecton socknr1 socknr2)
   returns a java-connection object
   socknr1 = socket where java process listen for commands
   socknr2 = socket where java sends commands for lisp"
  (let ((call-fd 0) (called-fd 0) (callback nil) (handler nil) (connection nil))
    (sleep 4)
    ;; Establish a socket connection on socketnr1 (10 tries before giving up)
    (setf call-fd
        (SOCKET::socket-connect socketnr1
                        "127.0.0.1"
                        :EXTERNAL-FORMAT (EXT::MAKE-ENCODING :CHARSET "ISO-8859-1"
                                           :LINE-TERMINATOR :UNIX)))
    ;; Give java time to accept the connection
    (sleep 4)
    ;; Establish a socket connection on socketnr1 (10 tries before giving up)
    (setf called-fd (SOCKET::socket-connect socketnr2
                      "127.0.0.1"
                             :EXTERNAL-FORMAT (EXT::MAKE-ENCODING :CHARSET "ISO-8859-1"
                                          :LINE-TERMINATOR :UNIX)))
    ;; Create now the java connection
    (setf connection (make-instance 'java-connection
                                    :call-in
                                    call-fd
                                    :call-out
                                    call-fd
                                    :called-in
                                    called-fd
                                    :called-out
                                    called-fd
                                    :callback
                                    nil
                                    :lispobjects
                                    (make-hash-table :test #'eql :size 100)
                                    :nrlispobjects
                                    0
                                    :java
                                    (make-instance 'java-object :id 0)))
    ;; Generate the handler for the input , must contain connection object
    (setf handler (generate-process-incoming connection))
    ;; Push the handler for the callback on the listener stack in CLISP
    (add-listener handler)
    ;; Make sure that the listeners are called
    (enable-listeners)
    (setf callback nil)
    ;; Not used by CLISP
    (setf (java-connection-handler connection) handler)
    (setf (java-connection-callback connection) callback)
    ;; Update connection
    (funcall handler)
    ;; return connection
    connection))


;; ACL version
#+:acl-socket
(defun make-java-connection (socketnr1 socketnr2)
  "(make-java-connecton socknr1 socknr2)
   returns a java-connection object
   socknr1 = socket where java process listen for commands
   socknr2 = socket where java sends commands for lisp"
  (let ((call-fd 0) (called-fd 0) (callback nil) (handler nil) (connection nil))
    (sleep 4)
    ;; Establish a socket connection on socketnr1 (10 tries before giving up)
    (setf call-fd (acl-socket::make-socket :remote-host "127.0.0.1" :remote-port socketnr1))
    ;; Give java time to accept the connection
    (sleep 4)
    ;; Establish a socket connection on socketnr1 (10 tries before giving up)
    (setf called-fd (acl-socket::make-socket :remote-host "127.0.0.1" :remote-port socketnr2))
    ;; Create now the java connection
    (setf connection (make-instance 'java-connection
                                    :call-in
                                    (extensions::make-fd-stream call-fd
                                                                :input
                                                                t
                                                                :buffering
                                                                :none)
                                    :call-out
                                    (extensions::make-fd-stream call-fd
                                                                :output
                                                                t
                                                                :buffering
                                                                :none)
                                    :called-in
                                    (extensions::make-fd-stream called-fd
                                                                :input
                                                                T
                                                                :buffering
                                                                :none)
                                    :called-out
                                    (extensions::make-fd-stream called-fd
                                                                :output
                                                                t
                                                                :buffering
                                                                :none)
                                    :callback
                                    nil
                                    :lispobjects
                                    (make-hash-table :test #'eql :size 100)
                                    :nrlispobjects
                                    0
                                    :java
                                    (make-instance 'java-object :id 0)))
    ;; Generate the handler for the input , must contain connection object
    (setf handler (generate-process-incoming connection))
    (setf callback (acl-socket::socket-os-fd called-fd))
    (setf callback (system:set-sigio-handler callback handler))
    (setf (java-connection-handler connection) handler)
    (setf (java-connection-callback connection) callback)
    ;; Update connection
    (funcall handler)
    ;; return connection
    connection))


;; *************************************************************************
;; Classes to specifify which base type java should use in case of integers
;; *************************************************************************

(defclass java-cast ()
  ((value :initarg :value :reader java-cast-value)
   (type :initarg :type :reader java-cast-type)))



;; ***************************************************************************
;; Extension of apply that works for functions and macros
;; ***************************************************************************

(defun general-apply (fun arg-lst)
  (handler-case
   (general-apply-1 fun arg-lst)
   (error (er) (message (format nil "Error : ~A" er)))))

(defmethod general-apply-1 ((fun symbol) arg-lst)
  (cond ((macro-function fun)
         (eval (cons fun arg-lst)))
        (t
         (apply fun arg-lst))))

(defmethod general-apply-1 ((fun function) arg-lst)
  (apply fun arg-lst))

(defmethod general-apply-1 ((fun t) arg-lst)
  (apply fun arg-lst))
      
;; ***************************************************************************
;; Generates a function to process incoming traffic from java
;; ***************************************************************************

(defun generate-process-incoming (connection)
  #'(lambda (&optional dummy &key ((:block block) nil))
            (let ((in (java-connection-called-in connection))
                  (out (java-connection-called-out connection))
                  (save-output *standard-output*))
              ;;    (setf *standard-output* (make-string-output-stream))
              (unwind-protect
               (loop
                (cond ((or (listen in) block) ;; If we may block during read or if there is input
                       (setf block nil)
                       (let ((command nil)
                             (type-command nil)
                             (result nil))
                         (setf type-command (read-char in nil nil))
                         (cond ((or (null type-command) ;; Java has closed connection
                                    (char= type-command #\A)) ;; Java called a abort
                                ;; We must abort the java connection
                                (delete-java-connection connection)
                                (return))
                               ((or (char= type-command #\C) (char= type-command #\c))
                                (when (char= type-command #\C) ;; Do not display output 
                                  (setf *standard-output* (make-string-output-stream)))
                                (do* ((arg-lst nil)
                                      (len (read in nil nil))
                                      (fun (call-java-method-result in
                                                                    connection))
                                      (i 1 (1+ i)))
                                  ((> i len)
                                   (unwind-protect ;; Ensure that we always send something back even if we have errors
                                    (progn (when (stringp fun)
                                             (setf fun (read-from-string fun)))                             
                                           (setf result (general-apply fun (reverse arg-lst))))
                                    (send-data-to-java result out connection)))
                                  (setf arg-lst
                                        (cons (call-java-method-result
                                               in
                                               connection)
                                              arg-lst))))
                               ((or (char= type-command #\S) (char= type-command #\s))
                                (when (char= type-command #\S)
                                  (setf *standard-output* (make-string-output-stream)))
                                ;; We get a string which must be evaluated
                                (setf command (call-java-method-result in connection))
                                (unwind-protect   ;; Ensure that we also send something back even in case of error
                                 (handler-case
                                  (setf result (eval (read-from-string command)))
                                  (error (er) (message (format nil "Error : ~A" er))))
                                 (send-data-to-java result out connection))))))
                      (t
                       (return))))
               (setf *standard-output* save-output)))))




;; **************************************************************************
;; Generate data to be handled by java as byte , integer , short , long
;; double or float
;; **************************************************************************

(defun java-byte (nr)
  (make-instance 'java-cast :type #\B :value nr))
(defun java-short (nr)
  (make-instance 'java-cast :type #\S :value nr))
(defun java-int (nr)
  (make-instance 'java-cast :type #\I :value nr))
(defun java-long (nr)
  (make-instance 'java-cast :type #\L :value nr))
(defun java-float (nr)
  (make-instance 'java-cast :type #\F :value nr))
(defun java-double (nr)
  (make-instance 'java-cast :type #\D :value nr))
(defun java-boolean (bool)
  (make-instance 'java-cast :type #\b :value bool))
(defun java-char (char)
  (make-instance 'java-cast :type #\C :value char))

;; **************************************************************************
;; Send data to java
;; **************************************************************************

(defmethod send-data-to-java ((cast java-cast) out connection)
  ;; We write a C folloew by te cast type to force java to do a cast
  ;; to correct numeric type
  (write-char #\C out)
  (write-char (java-cast-type cast) out)
  (send-data-to-java (java-cast-value cast) out connection))

(defmethod send-data-to-java ((nr number) out connection)
  ;; We first send a N and then a null teminated string containing the data
  (write-char #\N out)
  (princ nr out)
  (write-char (code-char 0) out)
  (force-output out))

(defmethod send-data-to-java ((bool (eql nil)) out connection)
  ;; We first send a B (to indicate a boolean type and then "F" to indicate a
  ;; false value
  (write-char #\B out)
  (write-char #\F out)
  (force-output out))

(defmethod send-data-to-java ((bool (eql T)) out connection)
  ;; We first send a B and then "T" to indicate a true value
  (write-char #\B out)
  (write-char #\T out)
  (force-output out))

(defmethod send-data-to-java ((arr array) out connection)
  ;; We send multidemnsional arrays as arrays of arrays
  (let* ((dim-lst (array-dimensions arr))
       (dim (first dim-lst)))
    (write-char #\A out) ;; Indicate data is a array
    (write dim :pretty nil :readably nil :escape nil :stream out)      ;; Write length of array
    (write-char (code-char 0) out)
    (force-output out)
    (do ((i 0 (1+ i)))
       ((>= i dim))
      (send-data-to-java-array (list i) (rest dim-lst) arr out connection))))

(defmethod send-data-to-java-array (rev-ind dim-lst arr out connection)
  (cond ((endp dim-lst)
   ;; No subdimensions , so we write out the entries
       (send-data-to-java (apply #'aref arr (reverse rev-ind)) out connection))
   (t
        ;; Subdimesnions , so we write arrays
        (let ((dim (first dim-lst)))
           (write-char #\A out)
          (write dim :pretty nil :readably nil :escape nil :stream out)
        (write-char (code-char 0) out)
     (force-output out)
      (do ((i 0 (1+ i)))
         ((>= i dim))
      (send-data-to-java-array (cons i rev-ind)
                            (rest dim-lst)
                                arr
                        out
                connection))))))

(defmethod send-data-to-java ((lst list) out connection)
  ;; Send a L follewed by a null terminated number to inidicate length of
  ;; the list follewed by the items in the list
  (write-char #\L out)
  (write (length lst) :pretty nil :readably nil :escape nil :stream out)
  (write-char (code-char 0) out)
  (force-output out)
  (dolist (el lst)
    (send-data-to-java el out connection)))

(defmethod send-data-to-java ((obj java-object) out connection)
  ;; Send a O followed by a id representing the object
  (write-char #\O out)
  (write (java-object-id obj) :pretty nil :readably nil :escape nil :stream out)
  (write-char (code-char 0) out)
  (force-output out))

(defmethod send-data-to-java ((result string) out connection)
  ;; Result is converted to a string , we send then a S then a null
  ;; terminated integer for the length of the string
  ;; followed by the string itself
  (write-char #\S out)
  (write (length result) :pretty nil :readably nil :escape nil :stream out)
  (write-char (code-char 0) out)
  ;;(write result :pretty nil :readably nil :escape nil :stream out)
  (write-string result out)
  (force-output out))

(defmethod send-data-to-java (result out connection)
  ;; Send LISP data not direct transferable in JAVA data
  (let ((nr (incf (java-connection-nrlispobjects connection))))
    (setf (gethash nr (java-connection-lispobjects connection)) result)
    (write-char #\l out)
    (send-data-to-java nr out connection)))

;; **************************************************************************
;; Delete a java connection (is LISP version specific
;; **************************************************************************
;; CMUCL version
#+:CMU
(defun delete-java-connection (c)
  (setf (java-connection-lispobjects c) nil)
  (system::remove-fd-handler (java-connection-callback c))
  (close (java-connection-called-in c))
  (close (java-connection-called-out c))
  (close (java-connection-call-in c))
  (close (java-connection-call-out c)))
;; CLISP version
#+CLISP
(defun delete-java-connection (c)
  (setf (java-connection-lispobjects c) nil)
  (remove-listener (java-connection-handler c))
  (when (not (has-listeners)) disable-listeners)
  (close (java-connection-called-in c))
  (close (java-connection-call-in c)))

;; ACL version
#+:acl-socket
(defun delete-java-connection (c)
  (setf (java-connection-lispobjects c) nil)
  (system::remove-sigio-handler (java-connection-callback c))
  (close (java-connection-called-in c))
  (close (java-connection-called-out c))
  (close (java-connection-call-in c))
  (close (java-connection-call-out c)))


;; *************************************************************************
;; Call a java method
;; *************************************************************************

(defmethod call-java-method (con (str string) (method string) &rest arg-list)
  (let ((out (java-connection-call-out con))
        (in (java-connection-call-in con)))
    (write-char #\C out) ;; Tell java that we call a static method of a class
    (send-data-to-java str out con) ;; Send the class name to java
    (send-data-to-java method out con) ;; Send the method name to java
    (send-data-to-java (length arg-list) out con) ;; Send the length of the arglist
    (dolist (arg arg-list (call-java-method-result in con))
            (send-data-to-java arg out con))))


(defmethod call-java-method (con (obj java-object) (method string) &rest arg-list)
  (let ((out (java-connection-call-out con))
       (in (java-connection-call-in con)))
    (write-char #\O out) ;; Tell java that we call a method of a object
    (send-data-to-java obj out con) ;; Send the object to java
    (send-data-to-java method out con) ;; Send method name to java
    (send-data-to-java (length arg-list) out con) ;; Send the length of the arglist
    (dolist (arg arg-list (call-java-method-result in con))
      (send-data-to-java arg out con))))

;; ***************************************************************************
;; Call a java method in the thread of the swing gui
;; ***************************************************************************

(defmethod call-java-method-swing (con (str string) (method string) &rest arg-list)
  (let ((out (java-connection-call-out con))
 (in (java-connection-call-in con)))
    (write-char #\S out) ;; Tell java that we call a static method of a class
    (send-data-to-java str out con) ;; Send the class name to java
    (send-data-to-java method out con) ;; Send the method name to java
    (send-data-to-java (length arg-list) out con) ;; Send the length of the arglist
    (dolist (arg arg-list (call-java-method-result in con))
      (send-data-to-java arg out con))))

;; **************************************************************************
;; Call a java method in the thread of the swing hui
;; **************************************************************************

(defmethod call-java-method-swing (con (obj java-object) (method string) &rest arg-list)
  (let ((out (java-connection-call-out con))
  (in (java-connection-call-in con)))
    (write-char #\s out) ;; Tell java that we call a method of a object
    (send-data-to-java obj out con) ;; Send the object to java
    (send-data-to-java method out con) ;; Send method name to java
    (send-data-to-java (length arg-list) out con) ;; Send the length of the arglist
    (dolist (arg arg-list (call-java-method-result in con))
      (send-data-to-java arg out con))))

;; ************************************************************************
;; Get a static member of a class
;; ************************************************************************

(defmethod get-java-member (con (str string) (member string))
  (let ((out (java-connection-call-out con))
        (in (java-connection-call-in con)))
    (write-char #\c out) ;; Tell java that we want a static member of a class
    (send-data-to-java str out con) ;; Send the class name to java
    (send-data-to-java member out con) ;; Send the member name to java
    (call-java-method-result in con))) ;; Get the result back

;; ************************************************************************
;; Get a member of a object
;; ************************************************************************

(defmethod get-java-member (con (obj java-object) (member string))
  (let ((out (java-connection-call-out con))
        (in (java-connection-call-in con)))
    (write-char #\o out) ;; Tell java that we want a member of the object
    (send-data-to-java obj out con) ;; Send the object to java
    (send-data-to-java member out con) ;; Send the member name to java
    (call-java-method-result in con))) ;; Get the result back

;; ************************************************************************
;; Create a java object
;; ************************************************************************

(defun make-java-object (con str &rest arg-list)
  (let ((out (java-connection-call-out con))
       (in (java-connection-call-in con)))
    (write-char #\N out) ;; Tell java that we want to create a object
    (send-data-to-java str out con) ;; Give java the class
    (send-data-to-java (length arg-list) out con) ;; Give nr of args to java
    (dolist (arg arg-list (call-java-method-result in con))
      (send-data-to-java arg out con))))


;; ************************************************************************
;; Get the result of a call to a java method
;; ************************************************************************

(defun call-java-method-result (in con)
  ;; Get the first character defining the type of result send back
  (let ((type-result (read-char in nil nil)))
    (declare (character type-result))
    (cond ((char= type-result #\B)
           ;; Boolean result is either T or NIL followed by a \n
           (read in nil nil))
          ((char= type-result #\N)
           ;; Numeric result type , just read the returned number
           (read in nil nil))
          ((char= type-result #\C)
           ;; Returned a character in the format #\d , just read the character
           (read in nil nil))
          ((char= type-result #\S) ;; Returned a string
           (do* ((max (read-positive-fixnum in)) (str (make-string max)) (i 0 (1+ i)))
             ((>= i max) str)
             (setf (char str i) (read-char in nil nil))))
          ((char= type-result #\O) ;; Returned a object
           (make-instance 'java-object :id (read-positive-fixnum in)))
          ((char= type-result #\L) ;; Returned a list
           (do ((max (read-positive-fixnum in)) (lst nil) (i 0 (1+ i)))
               ((>= i max) (reverse lst))
             (setf lst (cons (call-java-method-result in con) lst))))
          ((char= type-result #\l) ;; Returned a Lisp Object
           (gethash (read-positive-fixnum in) (java-connection-lispobjects con)))
          ((char= type-result #\A) ;; Returned a arrayof objects
           (do* ((max (read in nil nil)) (array (make-array max)) (i 0 (1+ i)))
                ((>= i max) array)
             (setf (aref array i) (call-java-method-result in con)))))))
;;; ***************************************************************************
;;; Get a positive fixnum from a stream
;;; ***************************************************************************
 
(defun read-positive-fixnum (strm)
  (declare (stream strm))
  (let ((result 0)
        (digit 0))
    (declare (fixnum result) (fixnum digit))
    (loop
     (setf digit (- (char-code (read-char strm NIL NIL)) 48)) ; Calculate the number
     (cond ((and (< -1 digit) (< digit 10))
        (setf result (+ (* result 10) digit)))
           (T
            (return result))))))










