;; *************************************************************************************** ;;
;; lisp debug v0.8  : source level debugger for lisp                                             ;;
;; Copyright (C) 1998 Marc Mertens                                                         ;;
;;                                                                                         ;;
;;     This program is free software; you can redistribute it and/or modify                ;;
;;    it under the terms of the GNU General Public License as published by                 ;;
;;    the Free Software Foundation; either version 2 of the License, or                    ;;
;;    (at your option) any later version.                                                  ;;
;;                                                                                         ;;
;;    This program is distributed in the hope that it will be useful,                      ;;
;;    but WITHOUT ANY WARRANTY; without even the implied warranty of                       ;;
;;    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                        ;;
;;    GNU General Public License for more details.                                         ;;
;;                                                                                         ;;
;;    You should have received a copy of the GNU General Public License                    ;;
;;    along with this program; if not, write to the Free Software                          ;;
;;    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA            ;;
;;                                                                                         ;;
;; Contact me on mmertens@akam.be                                                          ;;
;; ********************************************************************************************
;;
;; The following functions must be defined
;;
;; (get-unix-env "var" "default") ;; Get unix environment var, if not defined returns default
;; (get-syntax 'symbol)           ;; Get the syntaxc info of a symbol
;; (get-process-id)               ;; Get the process if of the lisp process
;; ********************************************************************************************

;; ***************************************************************************
;; Get unix environment variables
;; ***************************************************************************

(defun get-unix-env (var default)
  (cond ((rest (assoc (read-from-string (format nil ":~A" var)) COMMON-LISP-USER::*ENVIRONMENT-LIST*)))
   (t default)))

;; ***************************************************************************
;; Get the unix process id
;; ***************************************************************************
 
(defun get-process-id ()
  (unix:unix-getpid))
 
;; ***************************************************************************
;; Get the syntax of a call of a macro or symbol
;; ***************************************************************************

(defun get-syntax (symbol)
  (let ((type nil)
        (describe nil)
        (arglst nil)
        (ret nil))
    (multiple-value-setq (describe type) (get-syntax-describe symbol))
    (cond ((eq type 'function)
           (when (null (setf arglst (get-syntax-list-after-string "Function arguments:" describe)))
             (setf arglst (get-syntax-list-after-string "Its arguments are:" describe)))
           (setf arglst (get-syntax-check-arg arglst))
           (setf ret (get-syntax-list-after-string "Its result type is:" describe))
           (format nil "Function : ~A~%Arguments : ~A~%Return type : ~A"
                   symbol
                   (if arglst arglst "")
                   (if ret ret "")))
          ((eq type 'macro)
           (setf arglst (get-syntax-list-after-string "Macro arguments:" describe))
           (setf arglst (get-syntax-check-arg arglst))
           (setf ret (get-syntax-list-after-string "Its result type is:" describe))
           (format nil "Macro : ~A~%Arguments : ~A~%Return type : ~A"
                   symbol
                   (if arglst arglst)
                   (if ret ret "")))
          (t ""))))
      
;; *************************************************************************
;; Check ther correctness of a arglst (basically it has to start with a ())
;; *************************************************************************

(defun get-syntax-check-arg (arglst)
  (if (= (length arglst) 0)
      ""
    (if (char= (aref arglst 0) #\()
        arglst
      "")))

;; *************************************************************************
;; Run describe in the correct package to retrieve the description in
;; string form and the type of the symbol (macro or function)
;; (get-arglist-describe symbol)
;;      ==> description (string)
;;      ==> 'marco or 'function or nil
;; *************************************************************************

(defun get-syntax-describe (symbol)
  (let ((current-package (package-name *package*))
      (package nil)
       (name nil)
   (symb nil)
       (strm (make-string-output-stream)))
    (multiple-value-setq (package name) (get-syntax-split-symbol symbol))
    (cond (package
        (cond ((find-package package)
       (in-package (eval package))
       (setf symb (find-symbol (string-upcase name)))
                 (in-package (eval current-package))
                 (cond (symb
                      (ignore-errors (describe symb strm))
                   (values (get-output-stream-string strm)
               (get-syntax-type symb)))
            (t (values nil nil))))
      (t (values nil nil))))
   (t
      (setf symb (find-symbol (string-upcase name)))
           (cond (symb
                 (ignore-errors (describe symb strm))
               (values (get-output-stream-string strm)
        (get-syntax-type symb)))
          (t nil))))))

;; ************************************************************************
;; Get the type of a function
;; (get-syntax-type symbol) ==> 'function | 'macro | nil (no functin or macro)
;; ************************************************************************

(defun get-syntax-type (symbol)
  (if (fboundp symbol)
      (if (macro-function symbol)
       'macro
   'function)
    nil))

;; *************************************************************************
;; Find in a string the first lst or word after a given string
;; *************************************************************************

(defun get-syntax-list-after-string (match str)
  (let ((p (search match str))
   (l (length str)))
    (cond (p
      (setf p (+ (length match) p))
    ;; Find the first non white space character
    (do ((i p (1+ i)))
         ((or (>= i l)
           (not (or (char= (char str i) #\space)
                          (char= (char str i) #\newline)
                     (char= (char str i) #\tab)
                    (char= (char str i) #\return))))
         (setf p i)))
         (cond ((< p l)
       (cond ((char= (char str p) #\()
                   ;; Next word is a list
          (do ((i (1+ p) (1+ i))
                      (open 1))
                    ((or (zerop open) (>= i l))
                 (if (< i l)
                (subseq str p i)
                  nil))
            (cond ((char= (char str i) #\()
                    (incf open))
               ((char= (char str i) #\))
            (decf open)))))
               (t
                      ;; Next word is not a list
                 (do ((i p (1+ i)))
                ((or (>= i l)
            (char= (char str i) #\space)
                        (char= (char str i) #\newline)
            (char= (char str i) #\tab)
                (char= (char str i) #\return))
                     (if (< i l)
                        (subseq str p i)
                  nil))))))
        (t nil)))
         (t nil))))

;; *************************************************************************
;; Try to split a string representing a symbol in the package name and
;; the symbol name
;; "package::name" ==> ("package" "name")
;; "package:name"  ==> ("package" "name")
;; "name" ==> (NIL "name")
;; ************************************************************************

(defun get-syntax-split-symbol (symbol)
  (let ((pcolumn (position #\: symbol)))
    (cond (pcolumn
    (cond ((and (< pcolumn (1- (length symbol)))
                      (char= (char symbol (1+ pcolumn)) #\:))
               (values (subseq symbol 0 pcolumn)
                  (subseq symbol (+ pcolumn 2))))
        (t
             (values (subseq symbol 0 pcolumn)
                       (subseq symbol (+ pcolumn 1))))))
   (t (values NIL symbol)))))




;;; ****************************************************************************
;;; When CMUCL detects that Jabberwocky is started from the command line
;;; is starts talking to a STTY instead of to stdin,stderr ...
;;; This should solve this problem
;;; ****************************************************************************

(when (and (lisp::synonym-stream-p *terminal-io*)
	   (eq (lisp::synonym-stream-symbol *terminal-io*)
	       'SYSTEM::*TTY*))
  (setf *terminal-io* (make-two-way-stream system::*stdin* system::*stdout*)))
