;; *************************************************************************************** ;;
;; lisp debug v0.8  : source level debugger for lisp                                       ;;
;; Copyright (C) 1998 Marc Mertens                                                         ;;
;;                                                                                         ;;
;;     This program is free software; you can redistribute it and/or modify                ;;
;;    it under the terms of the GNU General Public License as published by                 ;;
;;    the Free Software Foundation; either version 2 of the License, or                    ;;
;;    (at your option) any later version.                                                  ;;
;;                                                                                         ;;
;;    This program is distributed in the hope that it will be useful,                      ;;
;;    but WITHOUT ANY WARRANTY; without even the implied warranty of                       ;;
;;    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                        ;;
;;    GNU General Public License for more details.                                         ;;
;;                                                                                         ;;
;;    You should have received a copy of the GNU General Public License                    ;;
;;    along with this program; if not, write to the Free Software                          ;;
;;    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA            ;;
;;                                                                                         ;;
;; Contact me on mmertens@akam.be                                                          ;;
;; ********************************************************************************************
;;
;; The following functions must be defined
;;
;; (get-unix-env "var" "default") ;; Get unix environment var, if not defined returns default
;; (get-syntax 'symbol)           ;; Get the syntaxc info of a symbol
;; (get-process-id)               ;; Get the process id of the lisp process
;; ********************************************************************************************

;; *****************************************************************************************
;; Using gray streams to process commands from the java process while still listening to the
;; input in the top level loop
;; *****************************************************************************************

;; Global variable to hold the standard input stream

(defconstant **save-standard-input** *standard-input*)

;; Define a special class for the modified input stream

(defclass debugger-input-stream (fundamental-input-stream) ())

;; Define the different methods needed for this class

(defmethod stream-read-char ((strm debugger-input-stream))
  (let ((chr nil))
    (loop
     (if (setf chr (read-char-no-hang **save-standard-input**))
         (return chr)
#+WIN32 (sleep 0.2)
#+UNIX  (sleep 0.02)
       ) ;; Sleep to avoid to much cpu use
     (DEBUGGER::call-listeners))))

(defmethod stream-unread-char ((strm debugger-input-stream) chr)
  (unread-char chr **save-standard-input**))

(defmethod stream-read-char-no-hang ((strm debugger-input-stream))
  (DEBUGGER::call-listeners)
  (if (listen **save-standard-input**)
      (read-char **save-standard-input**)
    nil))

(defmethod stream-listen ((strm debugger-input-stream))
  (DEBUGGER::call-listeners)
  (listen **save-standard-input**))

(defmethod stream-clear-input ((strm debugger-input-stream))
  (clear-input **save-standard-input**))


(in-package "DEBUGGER")

;; ***************************************************************************
;; (call-listeners) will call all the listeners during the spare moments in the
;; top level loop. The order of the call is the order of the listeners as they are
;; added.
;; (add-listener listener) will push the listener 'listener' on the stack
;; (remove-listener) will pop a listener from the stack
;; (has-listeners) will return T if listeners is non empty NIL otherwise
;; ***************************************************************************

(let ((listeners '()))
  (defun  call-listeners ()
    (mapc #'(lambda (listener) (funcall listener)) listeners))
  (defun has-listeners () (if listeners T NIL))
  (defun add-listener (listener) (push listener listeners))
  (defun remove-listener (listener) (setf listeners (remove listener listeners))))

;; ****************************************************************************
;; Enable/disable the listening to listeners
;; ****************************************************************************

(defun enable-listeners ()
  (setf *standard-input* (make-instance 'USER::debugger-input-stream)))
(defun disable-listeners ()
  (setf *standard-input* USER::**save-standard-input**))

;; ******************************************************************************************
;; Get unix system environment variable
;; ******************************************************************************************

(defun get-unix-env (var default)
  (cond ((system::getenv var))
        (t default)))

;; ****************************************************************************
;; Get the process ID of the LISP process 
;; ****************************************************************************

(defun get-process-id ()
  #-:WIN32 (SYSTEM::PROGRAM-ID))

;; ******************************************************************************************
;; Very primitive way of getting the syntax for a macro or symbol
;; ******************************************************************************************

(defun get-syntax (symbol-string)
  (let ((package nil)
       (name nil)
   (symbol nil))
    (multiple-value-setq (package name) (get-syntax-split-symbol symbol-string))
    (cond (package
           (setf symbol (find-symbol (string-upcase name) package)))
    (t
       (setf symbol (find-symbol (string-upcase name)))))
    (handler-case
   (if symbol
          (cond ((special-operator-p symbol) "Special operator")
        ((macro-function symbol)
                  (format nil "macro: ~A ~%Arguments: (?)" symbol))
           ((fboundp symbol)
        (format nil "function: ~A ~%Arguments: ~A" symbol (USER::arglist symbol)))
       (t "Unknown Syntax")))
      (error()))))

;; *************************************************************************
;; Try to split a string representing a symbol in the package name and
;; the symbol name
;; "package::name" ==> ("package" "name")
;; "package:name"  ==> ("package" "name")
;; "name" ==> (NIL "name")
;; ************************************************************************

(defun get-syntax-split-symbol (symbol)
  (let ((pcolumn (position #\: symbol)))
    (cond (pcolumn
        (cond ((and (< pcolumn (1- (length symbol)))
              (char= (char symbol (1+ pcolumn)) #\:))
       (values (subseq symbol 0 pcolumn)
              (subseq symbol (+ pcolumn 2))))
                (t
     (values (subseq symbol 0 pcolumn)
           (subseq symbol (+ pcolumn 1))))))
       (t (values NIL symbol)))))



    



