;;; time zone definition for non-UNIX systems
;;; Bruno Haible 8.-11.2.1994
;;; The data here comes from the /usr/lib/zoneinfo/ directory of the Linux
;;; package util-linux-1.5 and was constrained to the range 1950..2020.

(in-package "SYSTEM")

(defvar *default-time-zone*)
(defvar *default-dst-check*)

(labels
  ;; functions to check whether DST is in effect at a certain day and hour
  ;; in a certain march-based year.
  ((no-DST-p (Jahr Jahrtag Stunde) ; no DST at all
     (declare (ignore Jahr Jahrtag Stunde))
     nil
   )
   ;; Europe:
   (GB-Eire-DST-p (Jahr Jahrtag Stunde)
     (when (and (= Jahr 1967) (>= Jahrtag 337)) ; February 1968 ?
       (incf Jahr) (decf Jahrtag 366)
     )
     (and (<= 1960 Jahr 2020)
       (let ((Jahresstunde (+ (* 24 Jahrtag) Stunde))
             (Grenzen
               (svref '#((961 . 5160) ; 1960 : 10.4. 1h to 2.10. 1h
                         ; last Sunday in March 1h to last Sunday in October 1h
                         (601 . 5808) ; 1961 : 26.3. 1h to 29.10. 1h
                         (577 . 5784) ; 1962 : 25.3. 1h to 28.10. 1h
                         (721 . 5760) ; 1963 : 31.3. 1h to 27.10. 1h
                         ; Sunday>=19 in March 1h to last Sunday in October 1h
                         (505 . 5712) ; 1964 : 22.3. 1h to 25.10. 1h
                         (481 . 5688) ; 1965 : 21.3. 1h to 24.10. 1h
                         (457 . 5664) ; 1966 : 20.3. 1h to 23.10. 1h
                         (433 . 5808) ; 1967 : 19.3. 1h to 29.10. 1h
                         ; all year
                         (-263 . 8759) ; 1968 : 18.2. 1h to end of year
                         (0 . 8759) ; 1969 : all year
                         (0 . 8759) ; 1970 : all year
                         (0 . 5856) ; 1971 : beginning of year to 31.10. 1h
                         ; Sunday>=16 in March 1h to last Sunday in October 1h
                         (433 . 5808) ; 1972 : 19.3. 1h to 29.10. 1h
                         (409 . 5784) ; 1973 : 18.3. 1h to 28.10. 1h
                         (385 . 5760) ; 1974 : 17.3. 1h to 27.10. 1h
                         (361 . 5736) ; 1975 : 16.3. 1h to 26.10. 1h
                         (481 . 5688) ; 1976 : 21.3. 1h to 24.10. 1h
                         (457 . 5664) ; 1977 : 20.3. 1h to 23.10. 1h
                         (433 . 5808) ; 1978 : 19.3. 1h to 29.10. 1h
                         (409 . 5784) ; 1979 : 18.3. 1h to 28.10. 1h
                         (361 . 5736) ; 1980 : 16.3. 1h to 26.10. 1h
                         ; last Sunday in March 1h to last Sunday in October 1h
                         (673 . 5712) ; 1981 : 29.3. 1h to 25.10. 1h
                         (649 . 5688) ; 1982 : 28.3. 1h to 24.10. 1h
                         (625 . 5664) ; 1983 : 27.3. 1h to 23.10. 1h
                         (577 . 5784) ; 1984 : 25.3. 1h to 28.10. 1h
                         (721 . 5760) ; 1985 : 31.3. 1h to 27.10. 1h
                         (697 . 5736) ; 1986 : 30.3. 1h to 26.10. 1h
                         (673 . 5712) ; 1987 : 29.3. 1h to 25.10. 1h
                         (625 . 5664) ; 1988 : 27.3. 1h to 23.10. 1h
                         (601 . 5808) ; 1989 : 26.3. 1h to 29.10. 1h
                         (577 . 5784) ; 1990 : 25.3. 1h to 28.10. 1h
                         (721 . 5760) ; 1991 : 31.3. 1h to 27.10. 1h
                         (673 . 5712) ; 1992 : 29.3. 1h to 25.10. 1h
                         (649 . 5688) ; 1993 : 28.3. 1h to 24.10. 1h
                         (625 . 5664) ; 1994 : 27.3. 1h to 23.10. 1h
                         (601 . 5808) ; 1995 : 26.3. 1h to 29.10. 1h
                         (721 . 5760) ; 1996 : 31.3. 1h to 27.10. 1h
                         (697 . 5736) ; 1997 : 30.3. 1h to 26.10. 1h
                         (673 . 5712) ; 1998 : 29.3. 1h to 25.10. 1h
                         (649 . 5688) ; 1999 : 28.3. 1h to 24.10. 1h
                         (601 . 5808) ; 2000 : 26.3. 1h to 29.10. 1h
                         (577 . 5784) ; 2001 : 25.3. 1h to 28.10. 1h
                         (721 . 5760) ; 2002 : 31.3. 1h to 27.10. 1h
                         (697 . 5736) ; 2003 : 30.3. 1h to 26.10. 1h
                         (649 . 5688) ; 2004 : 28.3. 1h to 24.10. 1h
                         (625 . 5664) ; 2005 : 27.3. 1h to 23.10. 1h
                         (601 . 5808) ; 2006 : 26.3. 1h to 29.10. 1h
                         (577 . 5784) ; 2007 : 25.3. 1h to 28.10. 1h
                         (697 . 5736) ; 2008 : 30.3. 1h to 26.10. 1h
                         (673 . 5712) ; 2009 : 29.3. 1h to 25.10. 1h
                         (649 . 5688) ; 2010 : 28.3. 1h to 24.10. 1h
                         (625 . 5664) ; 2011 : 27.3. 1h to 23.10. 1h
                         (577 . 5784) ; 2012 : 25.3. 1h to 28.10. 1h
                         (721 . 5760) ; 2013 : 31.3. 1h to 27.10. 1h
                         (697 . 5736) ; 2014 : 30.3. 1h to 26.10. 1h
                         (673 . 5712) ; 2015 : 29.3. 1h to 25.10. 1h
                         (625 . 5664) ; 2016 : 27.3. 1h to 23.10. 1h
                         (601 . 5808) ; 2017 : 26.3. 1h to 29.10. 1h
                         (577 . 5784) ; 2018 : 25.3. 1h to 28.10. 1h
                         (721 . 5760) ; 2019 : 31.3. 1h to 27.10. 1h
                         (673 . 5712) ; 2020 : 29.3. 1h to 25.10. 1h
                        )
                      (- Jahr 1960)
            )) )
         (<= (car Grenzen) Jahresstunde (cdr Grenzen))
   ) ) )
   (West-Europe-DST-p (Jahr Jahrtag Stunde)
     (and (<= 1980 Jahr 2020)
       (let ((Jahresstunde (+ (* 24 Jahrtag) Stunde))
             (Grenzen
               (svref '#(; last Sunday in March 1h to last Sunday in September 1h
                         (697 . 5064) ; 1980 : 30.3. 1h to 28.9. 1h
                         (673 . 5040) ; 1981 : 29.3. 1h to 27.9. 1h
                         (649 . 5016) ; 1982 : 28.3. 1h to 26.9. 1h
                         (625 . 4992) ; 1983 : 27.3. 1h to 25.9. 1h
                         (577 . 5112) ; 1984 : 25.3. 1h to 30.9. 1h
                         (721 . 5088) ; 1985 : 31.3. 1h to 29.9. 1h
                         (697 . 5064) ; 1986 : 30.3. 1h to 28.9. 1h
                         (673 . 5040) ; 1987 : 29.3. 1h to 27.9. 1h
                         (625 . 4992) ; 1988 : 27.3. 1h to 25.9. 1h
                         (601 . 4968) ; 1989 : 26.3. 1h to 24.9. 1h
                         (577 . 5112) ; 1990 : 25.3. 1h to 30.9. 1h
                         (721 . 5088) ; 1991 : 31.3. 1h to 29.9. 1h
                         (673 . 5040) ; 1992 : 29.3. 1h to 27.9. 1h
                         (649 . 5016) ; 1993 : 28.3. 1h to 26.9. 1h
                         (625 . 4992) ; 1994 : 27.3. 1h to 25.9. 1h
                         (601 . 4968) ; 1995 : 26.3. 1h to 24.9. 1h
                         (721 . 5088) ; 1996 : 31.3. 1h to 29.9. 1h
                         (697 . 5064) ; 1997 : 30.3. 1h to 28.9. 1h
                         (673 . 5040) ; 1998 : 29.3. 1h to 27.9. 1h
                         (649 . 5016) ; 1999 : 28.3. 1h to 26.9. 1h
                         (601 . 4968) ; 2000 : 26.3. 1h to 24.9. 1h
                         (577 . 5112) ; 2001 : 25.3. 1h to 30.9. 1h
                         (721 . 5088) ; 2002 : 31.3. 1h to 29.9. 1h
                         (697 . 5064) ; 2003 : 30.3. 1h to 28.9. 1h
                         (649 . 5016) ; 2004 : 28.3. 1h to 26.9. 1h
                         (625 . 4992) ; 2005 : 27.3. 1h to 25.9. 1h
                         (601 . 4968) ; 2006 : 26.3. 1h to 24.9. 1h
                         (577 . 5112) ; 2007 : 25.3. 1h to 30.9. 1h
                         (697 . 5064) ; 2008 : 30.3. 1h to 28.9. 1h
                         (673 . 5040) ; 2009 : 29.3. 1h to 27.9. 1h
                         (649 . 5016) ; 2010 : 28.3. 1h to 26.9. 1h
                         (625 . 4992) ; 2011 : 27.3. 1h to 25.9. 1h
                         (577 . 5112) ; 2012 : 25.3. 1h to 30.9. 1h
                         (721 . 5088) ; 2013 : 31.3. 1h to 29.9. 1h
                         (697 . 5064) ; 2014 : 30.3. 1h to 28.9. 1h
                         (673 . 5040) ; 2015 : 29.3. 1h to 27.9. 1h
                         (625 . 4992) ; 2016 : 27.3. 1h to 25.9. 1h
                         (601 . 4968) ; 2017 : 26.3. 1h to 24.9. 1h
                         (577 . 5112) ; 2018 : 25.3. 1h to 30.9. 1h
                         (721 . 5088) ; 2019 : 31.3. 1h to 29.9. 1h
                         (673 . 5040) ; 2020 : 29.3. 1h to 27.9. 1h
                        )
                      (- Jahr 1980)
            )) )
         (<= (car Grenzen) Jahresstunde (cdr Grenzen))
   ) ) )
   (Mid-Europe-DST-p (Jahr Jahrtag Stunde)
     (and (<= 1980 Jahr 2020)
       (let ((Jahresstunde (+ (* 24 Jahrtag) Stunde))
             (Grenzen
               (svref '#(; I think DST was introduced in Germany in 1980.
                         ; last Sunday in March 2h to last Sunday in September 2h
                         (698 . 5065) ; 1980 : 30.3. 2h to 28.9. 2h
                         (674 . 5041) ; 1981 : 29.3. 2h to 27.9. 2h
                         (650 . 5017) ; 1982 : 28.3. 2h to 26.9. 2h
                         (626 . 4993) ; 1983 : 27.3. 2h to 25.9. 2h
                         (578 . 5113) ; 1984 : 25.3. 2h to 30.9. 2h
                         (722 . 5089) ; 1985 : 31.3. 2h to 29.9. 2h
                         (698 . 5065) ; 1986 : 30.3. 2h to 28.9. 2h
                         (674 . 5041) ; 1987 : 29.3. 2h to 27.9. 2h
                         (626 . 4993) ; 1988 : 27.3. 2h to 25.9. 2h
                         (602 . 4969) ; 1989 : 26.3. 2h to 24.9. 2h
                         (578 . 5113) ; 1990 : 25.3. 2h to 30.9. 2h
                         (722 . 5089) ; 1991 : 31.3. 2h to 29.9. 2h
                         (674 . 5041) ; 1992 : 29.3. 2h to 27.9. 2h
                         (650 . 5017) ; 1993 : 28.3. 2h to 26.9. 2h
                         (626 . 4993) ; 1994 : 27.3. 2h to 25.9. 2h
                         (602 . 4969) ; 1995 : 26.3. 2h to 24.9. 2h
                         ; last Sunday in March 2h to last Sunday in October 2h
                         (722 . 5761) ; 1996 : 31.3. 2h to 27.10. 2h
                         (698 . 5737) ; 1997 : 30.3. 2h to 26.10. 2h
                         (674 . 5713) ; 1998 : 29.3. 2h to 25.10. 2h
                         (650 . 5857) ; 1999 : 28.3. 2h to 31.10. 2h
                         (602 . 5809) ; 2000 : 26.3. 2h to 29.10. 2h
                         (578 . 5785) ; 2001 : 25.3. 2h to 28.10. 2h
                         (722 . 5761) ; 2002 : 31.3. 2h to 27.10. 2h
                         (698 . 5737) ; 2003 : 30.3. 2h to 26.10. 2h
                         (650 . 5857) ; 2004 : 28.3. 2h to 31.10. 2h
                         (626 . 5833) ; 2005 : 27.3. 2h to 30.10. 2h
                         (602 . 5809) ; 2006 : 26.3. 2h to 29.10. 2h
                         (578 . 5785) ; 2007 : 25.3. 2h to 28.10. 2h
                         (698 . 5737) ; 2008 : 30.3. 2h to 26.10. 2h
                         (674 . 5713) ; 2009 : 29.3. 2h to 25.10. 2h
                         (650 . 5857) ; 2010 : 28.3. 2h to 31.10. 2h
                         (626 . 5833) ; 2011 : 27.3. 2h to 30.10. 2h
                         (578 . 5785) ; 2012 : 25.3. 2h to 28.10. 2h
                         (722 . 5761) ; 2013 : 31.3. 2h to 27.10. 2h
                         (698 . 5737) ; 2014 : 30.3. 2h to 26.10. 2h
                         (674 . 5713) ; 2015 : 29.3. 2h to 25.10. 2h
                         (626 . 5833) ; 2016 : 27.3. 2h to 30.10. 2h
                         (602 . 5809) ; 2017 : 26.3. 2h to 29.10. 2h
                         (578 . 5785) ; 2018 : 25.3. 2h to 28.10. 2h
                         (722 . 5761) ; 2019 : 31.3. 2h to 27.10. 2h
                         (674 . 5713) ; 2020 : 29.3. 2h to 25.10. 2h
                        )
                      (- Jahr 1980)
            )) )
         (<= (car Grenzen) Jahresstunde (cdr Grenzen))
   ) ) )
   (East-Europe-DST-p (Jahr Jahrtag Stunde)
     (and (<= 1980 Jahr 2020)
       (let ((Jahresstunde (+ (* 24 Jahrtag) Stunde))
             (Grenzen
               (svref '#(; last Sunday in March 3h to last Sunday in September 3h
                         (699 . 5066) ; 1980 : 30.3. 3h to 28.9. 3h
                         (675 . 5042) ; 1981 : 29.3. 3h to 27.9. 3h
                         (651 . 5018) ; 1982 : 28.3. 3h to 26.9. 3h
                         (627 . 4994) ; 1983 : 27.3. 3h to 25.9. 3h
                         (579 . 5114) ; 1984 : 25.3. 3h to 30.9. 3h
                         (723 . 5090) ; 1985 : 31.3. 3h to 29.9. 3h
                         (699 . 5066) ; 1986 : 30.3. 3h to 28.9. 3h
                         (675 . 5042) ; 1987 : 29.3. 3h to 27.9. 3h
                         (627 . 4994) ; 1988 : 27.3. 3h to 25.9. 3h
                         (603 . 4970) ; 1989 : 26.3. 3h to 24.9. 3h
                         (579 . 5114) ; 1990 : 25.3. 3h to 30.9. 3h
                         (723 . 5090) ; 1991 : 31.3. 3h to 29.9. 3h
                         (675 . 5042) ; 1992 : 29.3. 3h to 27.9. 3h
                         (651 . 5018) ; 1993 : 28.3. 3h to 26.9. 3h
                         (627 . 4994) ; 1994 : 27.3. 3h to 25.9. 3h
                         (603 . 4970) ; 1995 : 26.3. 3h to 24.9. 3h
                         (723 . 5090) ; 1996 : 31.3. 3h to 29.9. 3h
                         (699 . 5066) ; 1997 : 30.3. 3h to 28.9. 3h
                         (675 . 5042) ; 1998 : 29.3. 3h to 27.9. 3h
                         (651 . 5018) ; 1999 : 28.3. 3h to 26.9. 3h
                         (603 . 4970) ; 2000 : 26.3. 3h to 24.9. 3h
                         (579 . 5114) ; 2001 : 25.3. 3h to 30.9. 3h
                         (723 . 5090) ; 2002 : 31.3. 3h to 29.9. 3h
                         (699 . 5066) ; 2003 : 30.3. 3h to 28.9. 3h
                         (651 . 5018) ; 2004 : 28.3. 3h to 26.9. 3h
                         (627 . 4994) ; 2005 : 27.3. 3h to 25.9. 3h
                         (603 . 4970) ; 2006 : 26.3. 3h to 24.9. 3h
                         (579 . 5114) ; 2007 : 25.3. 3h to 30.9. 3h
                         (699 . 5066) ; 2008 : 30.3. 3h to 28.9. 3h
                         (675 . 5042) ; 2009 : 29.3. 3h to 27.9. 3h
                         (651 . 5018) ; 2010 : 28.3. 3h to 26.9. 3h
                         (627 . 4994) ; 2011 : 27.3. 3h to 25.9. 3h
                         (579 . 5114) ; 2012 : 25.3. 3h to 30.9. 3h
                         (723 . 5090) ; 2013 : 31.3. 3h to 29.9. 3h
                         (699 . 5066) ; 2014 : 30.3. 3h to 28.9. 3h
                         (675 . 5042) ; 2015 : 29.3. 3h to 27.9. 3h
                         (627 . 4994) ; 2016 : 27.3. 3h to 25.9. 3h
                         (603 . 4970) ; 2017 : 26.3. 3h to 24.9. 3h
                         (579 . 5114) ; 2018 : 25.3. 3h to 30.9. 3h
                         (723 . 5090) ; 2019 : 31.3. 3h to 29.9. 3h
                         (675 . 5042) ; 2020 : 29.3. 3h to 27.9. 3h
                        )
                      (- Jahr 1980)
            )) )
         (<= (car Grenzen) Jahresstunde (cdr Grenzen))
   ) ) )
   ;; Asia:
   (Israel-DST-p (Jahr Jahrtag Stunde)
     (and (<= 1987 Jahr 2020)
       (let ((Jahresstunde (+ (* 24 Jahrtag) Stunde))
             (Grenzen
               (svref '#(; 3rd Sunday in April 2h to 4th Saturday in August 2h
                         (1178 . 4152) ; 1987 : 19.4. 2h to 21.8. 1h
                         (1130 . 4272) ; 1988 : 17.4. 2h to 26.8. 1h
                         (1106 . 4248) ; 1989 : 16.4. 2h to 25.8. 1h
                         (1082 . 4224) ; 1990 : 15.4. 2h to 24.8. 1h
                         (1226 . 4200) ; 1991 : 21.4. 2h to 23.8. 1h
                         (1178 . 4152) ; 1992 : 19.4. 2h to 21.8. 1h
                         (1154 . 4320) ; 1993 : 18.4. 2h to 28.8. 1h
                         (1130 . 4272) ; 1994 : 17.4. 2h to 26.8. 1h
                         (1106 . 4248) ; 1995 : 16.4. 2h to 25.8. 1h
                         (1226 . 4200) ; 1996 : 21.4. 2h to 23.8. 1h
                         (1202 . 4176) ; 1997 : 20.4. 2h to 22.8. 1h
                         (1178 . 4152) ; 1998 : 19.4. 2h to 21.8. 1h
                         (1154 . 4320) ; 1999 : 18.4. 2h to 28.8. 1h
                         (1106 . 4248) ; 2000 : 16.4. 2h to 25.8. 1h
                         (1082 . 4224) ; 2001 : 15.4. 2h to 24.8. 1h
                         (1226 . 4200) ; 2002 : 21.4. 2h to 23.8. 1h
                         (1202 . 4176) ; 2003 : 20.4. 2h to 22.8. 1h
                         (1154 . 4320) ; 2004 : 18.4. 2h to 28.8. 1h
                         (1130 . 4272) ; 2005 : 17.4. 2h to 26.8. 1h
                         (1106 . 4248) ; 2006 : 16.4. 2h to 25.8. 1h
                         (1082 . 4224) ; 2007 : 15.4. 2h to 24.8. 1h
                         (1202 . 4176) ; 2008 : 20.4. 2h to 22.8. 1h
                         (1178 . 4152) ; 2009 : 19.4. 2h to 21.8. 1h
                         (1154 . 4320) ; 2010 : 18.4. 2h to 28.8. 1h
                         (1130 . 4272) ; 2011 : 17.4. 2h to 26.8. 1h
                         (1082 . 4224) ; 2012 : 15.4. 2h to 24.8. 1h
                         (1226 . 4200) ; 2013 : 21.4. 2h to 23.8. 1h
                         (1202 . 4176) ; 2014 : 20.4. 2h to 22.8. 1h
                         (1178 . 4152) ; 2015 : 19.4. 2h to 21.8. 1h
                         (1130 . 4272) ; 2016 : 17.4. 2h to 26.8. 1h
                         (1106 . 4248) ; 2017 : 16.4. 2h to 25.8. 1h
                         (1082 . 4224) ; 2018 : 15.4. 2h to 24.8. 1h
                         (1226 . 4200) ; 2019 : 21.4. 2h to 23.8. 1h
                         (1178 . 4152) ; 2020 : 19.4. 2h to 21.8. 1h
                        )
                      (- Jahr 1987)
            )) )
         (<= (car Grenzen) Jahresstunde (cdr Grenzen))
   ) ) )
   (Iran-DST-p (Jahr Jahrtag Stunde)
     (and (<= 1988 Jahr 2020)
       (let ((Jahresstunde (+ (* 24 Jahrtag) Stunde))
             (Grenzen
               (svref '#(; last Sunday in March 2h to 3rd Sunday in September 2h
                         (626 . 4824) ; 1988 : 27.3. 2h to 18.9. 1h
                         (602 . 4800) ; 1989 : 26.3. 2h to 17.9. 1h
                         (578 . 4776) ; 1990 : 25.3. 2h to 16.9. 1h
                         (722 . 4752) ; 1991 : 31.3. 2h to 15.9. 1h
                         (674 . 4872) ; 1992 : 29.3. 2h to 20.9. 1h
                         (650 . 4848) ; 1993 : 28.3. 2h to 19.9. 1h
                         (626 . 4824) ; 1994 : 27.3. 2h to 18.9. 1h
                         (602 . 4800) ; 1995 : 26.3. 2h to 17.9. 1h
                         (722 . 4752) ; 1996 : 31.3. 2h to 15.9. 1h
                         (698 . 4896) ; 1997 : 30.3. 2h to 21.9. 1h
                         (674 . 4872) ; 1998 : 29.3. 2h to 20.9. 1h
                         (650 . 4848) ; 1999 : 28.3. 2h to 19.9. 1h
                         (602 . 4800) ; 2000 : 26.3. 2h to 17.9. 1h
                         (578 . 4776) ; 2001 : 25.3. 2h to 16.9. 1h
                         (722 . 4752) ; 2002 : 31.3. 2h to 15.9. 1h
                         (698 . 4896) ; 2003 : 30.3. 2h to 21.9. 1h
                         (650 . 4848) ; 2004 : 28.3. 2h to 19.9. 1h
                         (626 . 4824) ; 2005 : 27.3. 2h to 18.9. 1h
                         (602 . 4800) ; 2006 : 26.3. 2h to 17.9. 1h
                         (578 . 4776) ; 2007 : 25.3. 2h to 16.9. 1h
                         (698 . 4896) ; 2008 : 30.3. 2h to 21.9. 1h
                         (674 . 4872) ; 2009 : 29.3. 2h to 20.9. 1h
                         (650 . 4848) ; 2010 : 28.3. 2h to 19.9. 1h
                         (626 . 4824) ; 2011 : 27.3. 2h to 18.9. 1h
                         (578 . 4776) ; 2012 : 25.3. 2h to 16.9. 1h
                         (722 . 4752) ; 2013 : 31.3. 2h to 15.9. 1h
                         (698 . 4896) ; 2014 : 30.3. 2h to 21.9. 1h
                         (674 . 4872) ; 2015 : 29.3. 2h to 20.9. 1h
                         (626 . 4824) ; 2016 : 27.3. 2h to 18.9. 1h
                         (602 . 4800) ; 2017 : 26.3. 2h to 17.9. 1h
                         (578 . 4776) ; 2018 : 25.3. 2h to 16.9. 1h
                         (722 . 4752) ; 2019 : 31.3. 2h to 15.9. 1h
                         (674 . 4872) ; 2020 : 29.3. 2h to 20.9. 1h
                        )
                      (- Jahr 1988)
            )) )
         (<= (car Grenzen) Jahresstunde (cdr Grenzen))
   ) ) )
   (ROK-DST-p (Jahr Jahrtag Stunde)
     (and (<= 1987 Jahr 2020)
       (let ((Jahresstunde (+ (* 24 Jahrtag) Stunde))
             (Grenzen
               (svref '#(; 2nd Sunday in May 2h to 2nd Sunday in October 2h
                         (1682 . 5376) ; 1987 : 10.5. 2h to 11.10. 2h
                         (1634 . 5328) ; 1988 : 8.5. 2h to 9.10. 2h
                         (1778 . 5304) ; 1989 : 14.5. 2h to 8.10. 2h
                         (1754 . 5448) ; 1990 : 13.5. 2h to 14.10. 2h
                         (1730 . 5424) ; 1991 : 12.5. 2h to 13.10. 2h
                         (1682 . 5376) ; 1992 : 10.5. 2h to 11.10. 2h
                         (1658 . 5352) ; 1993 : 9.5. 2h to 10.10. 2h
                         (1634 . 5328) ; 1994 : 8.5. 2h to 9.10. 2h
                         (1778 . 5304) ; 1995 : 14.5. 2h to 8.10. 2h
                         (1730 . 5424) ; 1996 : 12.5. 2h to 13.10. 2h
                         (1706 . 5400) ; 1997 : 11.5. 2h to 12.10. 2h
                         (1682 . 5376) ; 1998 : 10.5. 2h to 11.10. 2h
                         (1658 . 5352) ; 1999 : 9.5. 2h to 10.10. 2h
                         (1778 . 5304) ; 2000 : 14.5. 2h to 8.10. 2h
                         (1754 . 5448) ; 2001 : 13.5. 2h to 14.10. 2h
                         (1730 . 5424) ; 2002 : 12.5. 2h to 13.10. 2h
                         (1706 . 5400) ; 2003 : 11.5. 2h to 12.10. 2h
                         (1658 . 5352) ; 2004 : 9.5. 2h to 10.10. 2h
                         (1634 . 5328) ; 2005 : 8.5. 2h to 9.10. 2h
                         (1778 . 5304) ; 2006 : 14.5. 2h to 8.10. 2h
                         (1754 . 5448) ; 2007 : 13.5. 2h to 14.10. 2h
                         (1706 . 5400) ; 2008 : 11.5. 2h to 12.10. 2h
                         (1682 . 5376) ; 2009 : 10.5. 2h to 11.10. 2h
                         (1658 . 5352) ; 2010 : 9.5. 2h to 10.10. 2h
                         (1634 . 5328) ; 2011 : 8.5. 2h to 9.10. 2h
                         (1754 . 5448) ; 2012 : 13.5. 2h to 14.10. 2h
                         (1730 . 5424) ; 2013 : 12.5. 2h to 13.10. 2h
                         (1706 . 5400) ; 2014 : 11.5. 2h to 12.10. 2h
                         (1682 . 5376) ; 2015 : 10.5. 2h to 11.10. 2h
                         (1634 . 5328) ; 2016 : 8.5. 2h to 9.10. 2h
                         (1778 . 5304) ; 2017 : 14.5. 2h to 8.10. 2h
                         (1754 . 5448) ; 2018 : 13.5. 2h to 14.10. 2h
                         (1730 . 5424) ; 2019 : 12.5. 2h to 13.10. 2h
                         (1682 . 5376) ; 2020 : 10.5. 2h to 11.10. 2h
                        )
                      (- Jahr 1987)
            )) )
         (<= (car Grenzen) Jahresstunde (cdr Grenzen))
   ) ) )
   ;; Australia:
   (Australia-Oz-DST-p (Jahr Jahrtag Stunde)
     (when (< Jahrtag 31) ; March?
       (setq Jahrtag (+ Jahrtag (- (UTag Jahr) (UTag (- Jahr 1))))
             Jahr (- Jahr 1)
     ) )
     (and (<= 1971 Jahr 2020)
       (let ((Jahresstunde (+ (* 24 Jahrtag) Stunde))
             (Grenzen
               (svref '#(; last Sunday in October 2h to 27.2. 2h
                         (5858 . 8713) ; 1971 : 31.10. 2h to 27.2. 2h
                         ; last Sunday in October 2h to first Sunday in March 2h
                         (5810 . 8833) ; 1972 : 29.10. 2h to 4.3. 2h
                         (5786 . 8809) ; 1973 : 28.10. 2h to 3.3. 2h
                         (5762 . 8785) ; 1974 : 27.10. 2h to 2.3. 2h
                         (5738 . 8929) ; 1975 : 26.10. 2h to 7.3. 2h
                         (5858 . 8881) ; 1976 : 31.10. 2h to 6.3. 2h
                         (5834 . 8857) ; 1977 : 30.10. 2h to 5.3. 2h
                         (5810 . 8833) ; 1978 : 29.10. 2h to 4.3. 2h
                         (5786 . 8809) ; 1979 : 28.10. 2h to 2.3. 2h
                         (5738 . 8761) ; 1980 : 26.10. 2h to 1.3. 2h
                         (5714 . 8905) ; 1981 : 25.10. 2h to 7.3. 2h
                         (5858 . 8881) ; 1982 : 31.10. 2h to 6.3. 2h
                         (5834 . 8857) ; 1983 : 30.10. 2h to 4.3. 2h
                         (5786 . 8809) ; 1984 : 28.10. 2h to 3.3. 2h
                         (5762 . 8785) ; 1985 : 27.10. 2h to 2.3. 2h
                         ; next-to-last Sunday in October 2h to third Sunday in March 2h
                         (5570 . 9097) ; 1986 : 19.10. 2h to 15.3. 2h
                         (5546 . 9241) ; 1987 : 18.10. 2h to 20.3. 2h
                         (5666 . 9193) ; 1988 : 23.10. 2h to 19.3. 2h
                         (5642 . 9169) ; 1989 : 22.10. 2h to 18.3. 2h
                         (5618 . 9145) ; 1990 : 21.10. 2h to 17.3. 2h
                         (5594 . 9121) ; 1991 : 20.10. 2h to 15.3. 2h
                         (5546 . 9241) ; 1992 : 18.10. 2h to 21.3. 2h
                         (5690 . 9217) ; 1993 : 24.10. 2h to 20.3. 2h
                         (5666 . 9193) ; 1994 : 23.10. 2h to 19.3. 2h
                         (5642 . 9169) ; 1995 : 22.10. 2h to 17.3. 2h
                         (5594 . 9121) ; 1996 : 20.10. 2h to 16.3. 2h
                         (5570 . 9097) ; 1997 : 19.10. 2h to 15.3. 2h
                         (5546 . 9241) ; 1998 : 18.10. 2h to 21.3. 2h
                         (5690 . 9217) ; 1999 : 24.10. 2h to 19.3. 2h
                         (5642 . 9169) ; 2000 : 22.10. 2h to 18.3. 2h
                         (5618 . 9145) ; 2001 : 21.10. 2h to 17.3. 2h
                         (5594 . 9121) ; 2002 : 20.10. 2h to 16.3. 2h
                         (5570 . 9265) ; 2003 : 19.10. 2h to 21.3. 2h
                         (5690 . 9217) ; 2004 : 24.10. 2h to 20.3. 2h
                         (5666 . 9193) ; 2005 : 23.10. 2h to 19.3. 2h
                         (5642 . 9169) ; 2006 : 22.10. 2h to 18.3. 2h
                         (5618 . 9145) ; 2007 : 21.10. 2h to 16.3. 2h
                         (5570 . 9097) ; 2008 : 19.10. 2h to 15.3. 2h
                         (5546 . 9241) ; 2009 : 18.10. 2h to 21.3. 2h
                         (5690 . 9217) ; 2010 : 24.10. 2h to 20.3. 2h
                         (5666 . 9193) ; 2011 : 23.10. 2h to 18.3. 2h
                         (5618 . 9145) ; 2012 : 21.10. 2h to 17.3. 2h
                         (5594 . 9121) ; 2013 : 20.10. 2h to 16.3. 2h
                         (5570 . 9097) ; 2014 : 19.10. 2h to 15.3. 2h
                         (5546 . 9241) ; 2015 : 18.10. 2h to 20.3. 2h
                         (5666 . 9193) ; 2016 : 23.10. 2h to 19.3. 2h
                         (5642 . 9169) ; 2017 : 22.10. 2h to 18.3. 2h
                         (5618 . 9145) ; 2018 : 21.10. 2h to 17.3. 2h
                         (5594 . 9121) ; 2019 : 20.10. 2h to 15.3. 2h
                         (5546 . 9241) ; 2020 : 18.10. 2h to 21.3. 2h
                        )
                      (- Jahr 1971)
            )) )
         (<= (car Grenzen) Jahresstunde (cdr Grenzen))
   ) ) )
   (Australia-Victoria-DST-p (Jahr Jahrtag Stunde)
     (when (< Jahrtag 31) ; March?
       (setq Jahrtag (+ Jahrtag (- (UTag Jahr) (UTag (- Jahr 1))))
             Jahr (- Jahr 1)
     ) )
     (and (<= 1971 Jahr 2020)
       (let ((Jahresstunde (+ (* 24 Jahrtag) Stunde))
             (Grenzen
               (svref '#(; last Sunday in October 2h to 27.2. 2h
                         (5858 . 8713) ; 1971 : 31.10. 2h to 27.2. 2h
                         ; last Sunday in October 2h to first Sunday in March 2h
                         (5810 . 8833) ; 1972 : 29.10. 2h to 4.3. 2h
                         (5786 . 8809) ; 1973 : 28.10. 2h to 3.3. 2h
                         (5762 . 8785) ; 1974 : 27.10. 2h to 2.3. 2h
                         (5738 . 8929) ; 1975 : 26.10. 2h to 7.3. 2h
                         (5858 . 8881) ; 1976 : 31.10. 2h to 6.3. 2h
                         (5834 . 8857) ; 1977 : 30.10. 2h to 5.3. 2h
                         (5810 . 8833) ; 1978 : 29.10. 2h to 4.3. 2h
                         (5786 . 8809) ; 1979 : 28.10. 2h to 2.3. 2h
                         (5738 . 8761) ; 1980 : 26.10. 2h to 1.3. 2h
                         (5714 . 8905) ; 1981 : 25.10. 2h to 7.3. 2h
                         (5858 . 8881) ; 1982 : 31.10. 2h to 6.3. 2h
                         (5834 . 8857) ; 1983 : 30.10. 2h to 4.3. 2h
                         (5786 . 8809) ; 1984 : 28.10. 2h to 3.3. 2h
                         (5762 . 9121) ; 1985 : 27.10. 2h to 16.3. 2h
                         ; next-to-last Sunday in October 2h to third Sunday in March 2h
                         (5570 . 9097) ; 1986 : 19.10. 2h to 15.3. 2h
                         (5546 . 9241) ; 1987 : 18.10. 2h to 20.3. 2h
                         (5666 . 9193) ; 1988 : 23.10. 2h to 19.3. 2h
                         (5642 . 9169) ; 1989 : 22.10. 2h to 18.3. 2h
                         (5618 . 9145) ; 1990 : 21.10. 2h to 17.3. 2h
                         (5594 . 9121) ; 1991 : 20.10. 2h to 15.3. 2h
                         (5546 . 9241) ; 1992 : 18.10. 2h to 21.3. 2h
                         (5690 . 9217) ; 1993 : 24.10. 2h to 20.3. 2h
                         (5666 . 9193) ; 1994 : 23.10. 2h to 19.3. 2h
                         (5642 . 9169) ; 1995 : 22.10. 2h to 17.3. 2h
                         (5594 . 9121) ; 1996 : 20.10. 2h to 16.3. 2h
                         (5570 . 9097) ; 1997 : 19.10. 2h to 15.3. 2h
                         (5546 . 9241) ; 1998 : 18.10. 2h to 21.3. 2h
                         (5690 . 9217) ; 1999 : 24.10. 2h to 19.3. 2h
                         (5642 . 9169) ; 2000 : 22.10. 2h to 18.3. 2h
                         (5618 . 9145) ; 2001 : 21.10. 2h to 17.3. 2h
                         (5594 . 9121) ; 2002 : 20.10. 2h to 16.3. 2h
                         (5570 . 9265) ; 2003 : 19.10. 2h to 21.3. 2h
                         (5690 . 9217) ; 2004 : 24.10. 2h to 20.3. 2h
                         (5666 . 9193) ; 2005 : 23.10. 2h to 19.3. 2h
                         (5642 . 9169) ; 2006 : 22.10. 2h to 18.3. 2h
                         (5618 . 9145) ; 2007 : 21.10. 2h to 16.3. 2h
                         (5570 . 9097) ; 2008 : 19.10. 2h to 15.3. 2h
                         (5546 . 9241) ; 2009 : 18.10. 2h to 21.3. 2h
                         (5690 . 9217) ; 2010 : 24.10. 2h to 20.3. 2h
                         (5666 . 9193) ; 2011 : 23.10. 2h to 18.3. 2h
                         (5618 . 9145) ; 2012 : 21.10. 2h to 17.3. 2h
                         (5594 . 9121) ; 2013 : 20.10. 2h to 16.3. 2h
                         (5570 . 9097) ; 2014 : 19.10. 2h to 15.3. 2h
                         (5546 . 9241) ; 2015 : 18.10. 2h to 20.3. 2h
                         (5666 . 9193) ; 2016 : 23.10. 2h to 19.3. 2h
                         (5642 . 9169) ; 2017 : 22.10. 2h to 18.3. 2h
                         (5618 . 9145) ; 2018 : 21.10. 2h to 17.3. 2h
                         (5594 . 9121) ; 2019 : 20.10. 2h to 15.3. 2h
                         (5546 . 9241) ; 2020 : 18.10. 2h to 21.3. 2h
                        )
                      (- Jahr 1971)
            )) )
         (<= (car Grenzen) Jahresstunde (cdr Grenzen))
   ) ) )
   (Australia-NSW-DST-p (Jahr Jahrtag Stunde)
     (when (< Jahrtag 61) ; March or April?
       (setq Jahrtag (+ Jahrtag (- (UTag Jahr) (UTag (- Jahr 1))))
             Jahr (- Jahr 1)
     ) )
     (and (<= 1971 Jahr 2020)
       (let ((Jahresstunde (+ (* 24 Jahrtag) Stunde))
             (Grenzen
               (svref '#(; last Sunday in October 2h to 27.2. 2h
                         (5858 . 8713) ; 1971 : 31.10. 2h to 27.2. 2h
                         ; last Sunday in October 2h to first Sunday in March 2h
                         (5810 . 8833) ; 1972 : 29.10. 2h to 4.3. 2h
                         (5786 . 8809) ; 1973 : 28.10. 2h to 3.3. 2h
                         (5762 . 8785) ; 1974 : 27.10. 2h to 2.3. 2h
                         (5738 . 8929) ; 1975 : 26.10. 2h to 7.3. 2h
                         (5858 . 8881) ; 1976 : 31.10. 2h to 6.3. 2h
                         (5834 . 8857) ; 1977 : 30.10. 2h to 5.3. 2h
                         (5810 . 8833) ; 1978 : 29.10. 2h to 4.3. 2h
                         (5786 . 8809) ; 1979 : 28.10. 2h to 2.3. 2h
                         (5738 . 8761) ; 1980 : 26.10. 2h to 1.3. 2h
                         (5714 . 9577) ; 1981 : 25.10. 2h to 4.4. 2h
                         (5858 . 8881) ; 1982 : 31.10. 2h to 6.3. 2h
                         (5834 . 8857) ; 1983 : 30.10. 2h to 4.3. 2h
                         (5786 . 8809) ; 1984 : 28.10. 2h to 3.3. 2h
                         (5762 . 9121) ; 1985 : 27.10. 2h to 16.3. 2h
                         ; next-to-last Sunday in October 2h to third Sunday in March 2h
                         (5570 . 9097) ; 1986 : 19.10. 2h to 15.3. 2h
                         ; last Sunday in October 2h to third Sunday in March 2h
                         (5714 . 9241) ; 1987 : 25.10. 2h to 20.3. 2h
                         (5834 . 9193) ; 1988 : 30.10. 2h to 19.3. 2h
                         (5810 . 9169) ; 1989 : 29.10. 2h to 18.3. 2h
                         (5786 . 9145) ; 1990 : 28.10. 2h to 17.3. 2h
                         (5762 . 9121) ; 1991 : 27.10. 2h to 15.3. 2h
                         (5714 . 9241) ; 1992 : 25.10. 2h to 21.3. 2h
                         (5858 . 9217) ; 1993 : 31.10. 2h to 20.3. 2h
                         (5834 . 9193) ; 1994 : 30.10. 2h to 19.3. 2h
                         (5810 . 9169) ; 1995 : 29.10. 2h to 17.3. 2h
                         (5762 . 9121) ; 1996 : 27.10. 2h to 16.3. 2h
                         (5738 . 9097) ; 1997 : 26.10. 2h to 15.3. 2h
                         (5714 . 9241) ; 1998 : 25.10. 2h to 21.3. 2h
                         (5858 . 9217) ; 1999 : 31.10. 2h to 19.3. 2h
                         (5810 . 9169) ; 2000 : 29.10. 2h to 18.3. 2h
                         (5786 . 9145) ; 2001 : 28.10. 2h to 17.3. 2h
                         (5762 . 9121) ; 2002 : 27.10. 2h to 16.3. 2h
                         (5738 . 9265) ; 2003 : 26.10. 2h to 21.3. 2h
                         (5858 . 9217) ; 2004 : 31.10. 2h to 20.3. 2h
                         (5834 . 9193) ; 2005 : 30.10. 2h to 19.3. 2h
                         (5810 . 9169) ; 2006 : 29.10. 2h to 18.3. 2h
                         (5786 . 9145) ; 2007 : 28.10. 2h to 16.3. 2h
                         (5738 . 9097) ; 2008 : 26.10. 2h to 15.3. 2h
                         (5714 . 9241) ; 2009 : 25.10. 2h to 21.3. 2h
                         (5858 . 9217) ; 2010 : 31.10. 2h to 20.3. 2h
                         (5834 . 9193) ; 2011 : 30.10. 2h to 18.3. 2h
                         (5786 . 9145) ; 2012 : 28.10. 2h to 17.3. 2h
                         (5762 . 9121) ; 2013 : 27.10. 2h to 16.3. 2h
                         (5738 . 9097) ; 2014 : 26.10. 2h to 15.3. 2h
                         (5714 . 9241) ; 2015 : 25.10. 2h to 20.3. 2h
                         (5834 . 9193) ; 2016 : 30.10. 2h to 19.3. 2h
                         (5810 . 9169) ; 2017 : 29.10. 2h to 18.3. 2h
                         (5786 . 9145) ; 2018 : 28.10. 2h to 17.3. 2h
                         (5762 . 9121) ; 2019 : 27.10. 2h to 15.3. 2h
                         (5714 . 9241) ; 2020 : 25.10. 2h to 21.3. 2h
                        )
                      (- Jahr 1971)
            )) )
         (<= (car Grenzen) Jahresstunde (cdr Grenzen))
   ) ) )
   (NZ-DST-p (Jahr Jahrtag Stunde)
     (when (< Jahrtag 31) ; March?
       (setq Jahrtag (+ Jahrtag (- (UTag Jahr) (UTag (- Jahr 1))))
             Jahr (- Jahr 1)
     ) )
     (and (<= 1974 Jahr 2020)
       (let ((Jahresstunde (+ (* 24 Jahrtag) Stunde))
             (Grenzen
               (svref '#(; last Sunday in October 2h to first Sunday in March 2h
                         (5762 . 8785) ; 1974 : 27.10. 2h to 2.3. 2h
                         (5738 . 8929) ; 1975 : 26.10. 2h to 7.3. 2h
                         (5858 . 8881) ; 1976 : 31.10. 2h to 6.3. 2h
                         (5834 . 8857) ; 1977 : 30.10. 2h to 5.3. 2h
                         (5810 . 8833) ; 1978 : 29.10. 2h to 4.3. 2h
                         (5786 . 8809) ; 1979 : 28.10. 2h to 2.3. 2h
                         (5738 . 8761) ; 1980 : 26.10. 2h to 1.3. 2h
                         (5714 . 8905) ; 1981 : 25.10. 2h to 7.3. 2h
                         (5858 . 8881) ; 1982 : 31.10. 2h to 6.3. 2h
                         (5834 . 8857) ; 1983 : 30.10. 2h to 4.3. 2h
                         (5786 . 8809) ; 1984 : 28.10. 2h to 3.3. 2h
                         (5762 . 8785) ; 1985 : 27.10. 2h to 2.3. 2h
                         (5738 . 8761) ; 1986 : 26.10. 2h to 1.3. 2h
                         (5714 . 8905) ; 1987 : 25.10. 2h to 6.3. 2h
                         (5834 . 8857) ; 1988 : 30.10. 2h to 5.3. 2h
                         (5810 . 8833) ; 1989 : 29.10. 2h to 4.3. 2h
                         (5786 . 8809) ; 1990 : 28.10. 2h to 3.3. 2h
                         (5762 . 8785) ; 1991 : 27.10. 2h to 1.3. 2h
                         (5714 . 8905) ; 1992 : 25.10. 2h to 7.3. 2h
                         (5858 . 8881) ; 1993 : 31.10. 2h to 6.3. 2h
                         (5834 . 8857) ; 1994 : 30.10. 2h to 5.3. 2h
                         (5810 . 8833) ; 1995 : 29.10. 2h to 3.3. 2h
                         (5762 . 8785) ; 1996 : 27.10. 2h to 2.3. 2h
                         (5738 . 8761) ; 1997 : 26.10. 2h to 1.3. 2h
                         (5714 . 8905) ; 1998 : 25.10. 2h to 7.3. 2h
                         (5858 . 8881) ; 1999 : 31.10. 2h to 5.3. 2h
                         (5810 . 8833) ; 2000 : 29.10. 2h to 4.3. 2h
                         (5786 . 8809) ; 2001 : 28.10. 2h to 3.3. 2h
                         (5762 . 8785) ; 2002 : 27.10. 2h to 2.3. 2h
                         (5738 . 8929) ; 2003 : 26.10. 2h to 7.3. 2h
                         (5858 . 8881) ; 2004 : 31.10. 2h to 6.3. 2h
                         (5834 . 8857) ; 2005 : 30.10. 2h to 5.3. 2h
                         (5810 . 8833) ; 2006 : 29.10. 2h to 4.3. 2h
                         (5786 . 8809) ; 2007 : 28.10. 2h to 2.3. 2h
                         (5738 . 8761) ; 2008 : 26.10. 2h to 1.3. 2h
                         (5714 . 8905) ; 2009 : 25.10. 2h to 7.3. 2h
                         (5858 . 8881) ; 2010 : 31.10. 2h to 6.3. 2h
                         (5834 . 8857) ; 2011 : 30.10. 2h to 4.3. 2h
                         (5786 . 8809) ; 2012 : 28.10. 2h to 3.3. 2h
                         (5762 . 8785) ; 2013 : 27.10. 2h to 2.3. 2h
                         (5738 . 8761) ; 2014 : 26.10. 2h to 1.3. 2h
                         (5714 . 8905) ; 2015 : 25.10. 2h to 6.3. 2h
                         (5834 . 8857) ; 2016 : 30.10. 2h to 5.3. 2h
                         (5810 . 8833) ; 2017 : 29.10. 2h to 4.3. 2h
                         (5786 . 8809) ; 2018 : 28.10. 2h to 3.3. 2h
                         (5762 . 8785) ; 2019 : 27.10. 2h to 1.3. 2h
                         (5714 . 8905) ; 2020 : 25.10. 2h to 7.3. 2h
                        )
                      (- Jahr 1974)
            )) )
         (<= (car Grenzen) Jahresstunde (cdr Grenzen))
   ) ) )
   ;; Africa:
   (Egypt-DST-p (Jahr Jahrtag Stunde)
     (and (>= Jahr 1960) (not (<= 1981 Jahr 1982))
          ; 1.5. 2h to 1.10. 1h
          (let ((Jahresstunde (+ (* 24 Jahrtag) Stunde)))
            (<= 1466 Jahresstunde 5136)
   ) )    )
   (Libya-DST-p (Jahr Jahrtag Stunde)
     (and (>= Jahr 1982)
          ; 1.4. 2h to 30.9. 1h
          (let ((Jahresstunde (+ (* 24 Jahrtag) Stunde)))
            (<= 746 Jahresstunde 5112)
   ) )    )
   ;; North America:
   (Canada-DST-p (Jahr Jahrtag Stunde)
     (and (<= 1969 Jahr 2020)
       (let ((Jahresstunde (+ (* 24 Jahrtag) Stunde))
             (Grenzen
               (svref '#(; last Sunday in April 2h to last Sunday in October 2h
                         (1370 . 5736) ; 1969 : 27.4. 2h to 26.10. 1h
                         (1346 . 5712) ; 1970 : 26.4. 2h to 25.10. 1h
                         (1322 . 5856) ; 1971 : 25.4. 2h to 31.10. 1h
                         (1442 . 5808) ; 1972 : 30.4. 2h to 29.10. 1h
                         (1418 . 5784) ; 1973 : 29.4. 2h to 28.10. 1h
                         (1394 . 5760) ; 1974 : 28.4. 2h to 27.10. 1h
                         (1370 . 5736) ; 1975 : 27.4. 2h to 26.10. 1h
                         (1322 . 5856) ; 1976 : 25.4. 2h to 31.10. 1h
                         (1298 . 5832) ; 1977 : 24.4. 2h to 30.10. 1h
                         (1442 . 5808) ; 1978 : 30.4. 2h to 29.10. 1h
                         (1418 . 5784) ; 1979 : 29.4. 2h to 28.10. 1h
                         (1370 . 5736) ; 1980 : 27.4. 2h to 26.10. 1h
                         (1346 . 5712) ; 1981 : 26.4. 2h to 25.10. 1h
                         (1322 . 5856) ; 1982 : 25.4. 2h to 31.10. 1h
                         (1298 . 5832) ; 1983 : 24.4. 2h to 30.10. 1h
                         (1418 . 5784) ; 1984 : 29.4. 2h to 28.10. 1h
                         (1394 . 5760) ; 1985 : 28.4. 2h to 27.10. 1h
                         (1370 . 5736) ; 1986 : 27.4. 2h to 26.10. 1h
                         ; first Sunday in April 2h to last Sunday in October 2h
                         (842 . 5712) ; 1987 : 5.4. 2h to 25.10. 1h
                         (794 . 5832) ; 1988 : 3.4. 2h to 30.10. 1h
                         (770 . 5808) ; 1989 : 2.4. 2h to 29.10. 1h
                         (746 . 5784) ; 1990 : 1.4. 2h to 28.10. 1h
                         (890 . 5760) ; 1991 : 7.4. 2h to 27.10. 1h
                         (842 . 5712) ; 1992 : 5.4. 2h to 25.10. 1h
                         (818 . 5856) ; 1993 : 4.4. 2h to 31.10. 1h
                         (794 . 5832) ; 1994 : 3.4. 2h to 30.10. 1h
                         (770 . 5808) ; 1995 : 2.4. 2h to 29.10. 1h
                         (890 . 5760) ; 1996 : 7.4. 2h to 27.10. 1h
                         (866 . 5736) ; 1997 : 6.4. 2h to 26.10. 1h
                         (842 . 5712) ; 1998 : 5.4. 2h to 25.10. 1h
                         (818 . 5856) ; 1999 : 4.4. 2h to 31.10. 1h
                         (770 . 5808) ; 2000 : 2.4. 2h to 29.10. 1h
                         (746 . 5784) ; 2001 : 1.4. 2h to 28.10. 1h
                         (890 . 5760) ; 2002 : 7.4. 2h to 27.10. 1h
                         (866 . 5736) ; 2003 : 6.4. 2h to 26.10. 1h
                         (818 . 5856) ; 2004 : 4.4. 2h to 31.10. 1h
                         (794 . 5832) ; 2005 : 3.4. 2h to 30.10. 1h
                         (770 . 5808) ; 2006 : 2.4. 2h to 29.10. 1h
                         (746 . 5784) ; 2007 : 1.4. 2h to 28.10. 1h
                         (866 . 5736) ; 2008 : 6.4. 2h to 26.10. 1h
                         (842 . 5712) ; 2009 : 5.4. 2h to 25.10. 1h
                         (818 . 5856) ; 2010 : 4.4. 2h to 31.10. 1h
                         (794 . 5832) ; 2011 : 3.4. 2h to 30.10. 1h
                         (746 . 5784) ; 2012 : 1.4. 2h to 28.10. 1h
                         (890 . 5760) ; 2013 : 7.4. 2h to 27.10. 1h
                         (866 . 5736) ; 2014 : 6.4. 2h to 26.10. 1h
                         (842 . 5712) ; 2015 : 5.4. 2h to 25.10. 1h
                         (794 . 5832) ; 2016 : 3.4. 2h to 30.10. 1h
                         (770 . 5808) ; 2017 : 2.4. 2h to 29.10. 1h
                         (746 . 5784) ; 2018 : 1.4. 2h to 28.10. 1h
                         (890 . 5760) ; 2019 : 7.4. 2h to 27.10. 1h
                         (842 . 5712) ; 2020 : 5.4. 2h to 25.10. 1h
                        )
                      (- Jahr 1969)
            )) )
         (<= (car Grenzen) Jahresstunde (cdr Grenzen))
   ) ) )
   (Cuba-DST-p (Jahr Jahrtag Stunde)
     (and (<= 1979 Jahr 2020)
       (let ((Jahresstunde (+ (* 24 Jahrtag) Stunde))
             (Grenzen
               (svref '#(; third Sunday in March 0h to 2nd Sunday in October 0h
                         (408 . 5446) ; 1979 : 18.3. 0h to 13.10. 23h
                         (360 . 5398) ; 1980 : 16.3. 0h to 11.10. 23h
                         ; 2nd Sunday in May 0h to 2nd Sunday in October 0h
                         (1680 . 5374) ; 1981 : 10.5. 0h to 10.10. 23h
                         (1656 . 5350) ; 1982 : 9.5. 0h to 9.10. 23h
                         (1632 . 5326) ; 1983 : 8.5. 0h to 8.10. 23h
                         (1752 . 5446) ; 1984 : 13.5. 0h to 13.10. 23h
                         (1728 . 5422) ; 1985 : 12.5. 0h to 12.10. 23h
                         (1704 . 5398) ; 1986 : 11.5. 0h to 11.10. 23h
                         (1680 . 5374) ; 1987 : 10.5. 0h to 10.10. 23h
                         (1632 . 5326) ; 1988 : 8.5. 0h to 8.10. 23h
                         (1776 . 5302) ; 1989 : 14.5. 0h to 7.10. 23h
                         (1752 . 5446) ; 1990 : 13.5. 0h to 13.10. 23h
                         (1728 . 5422) ; 1991 : 12.5. 0h to 12.10. 23h
                         (1680 . 5374) ; 1992 : 10.5. 0h to 10.10. 23h
                         (1656 . 5350) ; 1993 : 9.5. 0h to 9.10. 23h
                         (1632 . 5326) ; 1994 : 8.5. 0h to 8.10. 23h
                         (1776 . 5302) ; 1995 : 14.5. 0h to 7.10. 23h
                         (1728 . 5422) ; 1996 : 12.5. 0h to 12.10. 23h
                         (1704 . 5398) ; 1997 : 11.5. 0h to 11.10. 23h
                         (1680 . 5374) ; 1998 : 10.5. 0h to 10.10. 23h
                         (1656 . 5350) ; 1999 : 9.5. 0h to 9.10. 23h
                         (1776 . 5302) ; 2000 : 14.5. 0h to 7.10. 23h
                         (1752 . 5446) ; 2001 : 13.5. 0h to 13.10. 23h
                         (1728 . 5422) ; 2002 : 12.5. 0h to 12.10. 23h
                         (1704 . 5398) ; 2003 : 11.5. 0h to 11.10. 23h
                         (1656 . 5350) ; 2004 : 9.5. 0h to 9.10. 23h
                         (1632 . 5326) ; 2005 : 8.5. 0h to 8.10. 23h
                         (1776 . 5302) ; 2006 : 14.5. 0h to 7.10. 23h
                         (1752 . 5446) ; 2007 : 13.5. 0h to 13.10. 23h
                         (1704 . 5398) ; 2008 : 11.5. 0h to 11.10. 23h
                         (1680 . 5374) ; 2009 : 10.5. 0h to 10.10. 23h
                         (1656 . 5350) ; 2010 : 9.5. 0h to 9.10. 23h
                         (1632 . 5326) ; 2011 : 8.5. 0h to 8.10. 23h
                         (1752 . 5446) ; 2012 : 13.5. 0h to 13.10. 23h
                         (1728 . 5422) ; 2013 : 12.5. 0h to 12.10. 23h
                         (1704 . 5398) ; 2014 : 11.5. 0h to 11.10. 23h
                         (1680 . 5374) ; 2015 : 10.5. 0h to 10.10. 23h
                         (1632 . 5326) ; 2016 : 8.5. 0h to 8.10. 23h
                         (1776 . 5302) ; 2017 : 14.5. 0h to 7.10. 23h
                         (1752 . 5446) ; 2018 : 13.5. 0h to 13.10. 23h
                         (1728 . 5422) ; 2019 : 12.5. 0h to 12.10. 23h
                         (1680 . 5374) ; 2020 : 10.5. 0h to 10.10. 23h
                        )
                      (- Jahr 1979)
            )) )
         (<= (car Grenzen) Jahresstunde (cdr Grenzen))
   ) ) )
   (Mexico-DST-p (Jahr Jahrtag Stunde)
     (and (<= 1987 Jahr 2020)
       (let ((Jahresstunde (+ (* 24 Jahrtag) Stunde))
             (Grenzen
               (svref '#(; first Sunday in April 2h to last Sunday in October 2h
                         (842 . 5712) ; 1987 : 5.4. 2h to 25.10. 1h
                         (794 . 5832) ; 1988 : 3.4. 2h to 30.10. 1h
                         (770 . 5808) ; 1989 : 2.4. 2h to 29.10. 1h
                         (746 . 5784) ; 1990 : 1.4. 2h to 28.10. 1h
                         (890 . 5760) ; 1991 : 7.4. 2h to 27.10. 1h
                         (842 . 5712) ; 1992 : 5.4. 2h to 25.10. 1h
                         (818 . 5856) ; 1993 : 4.4. 2h to 31.10. 1h
                         (794 . 5832) ; 1994 : 3.4. 2h to 30.10. 1h
                         (770 . 5808) ; 1995 : 2.4. 2h to 29.10. 1h
                         (890 . 5760) ; 1996 : 7.4. 2h to 27.10. 1h
                         (866 . 5736) ; 1997 : 6.4. 2h to 26.10. 1h
                         (842 . 5712) ; 1998 : 5.4. 2h to 25.10. 1h
                         (818 . 5856) ; 1999 : 4.4. 2h to 31.10. 1h
                         (770 . 5808) ; 2000 : 2.4. 2h to 29.10. 1h
                         (746 . 5784) ; 2001 : 1.4. 2h to 28.10. 1h
                         (890 . 5760) ; 2002 : 7.4. 2h to 27.10. 1h
                         (866 . 5736) ; 2003 : 6.4. 2h to 26.10. 1h
                         (818 . 5856) ; 2004 : 4.4. 2h to 31.10. 1h
                         (794 . 5832) ; 2005 : 3.4. 2h to 30.10. 1h
                         (770 . 5808) ; 2006 : 2.4. 2h to 29.10. 1h
                         (746 . 5784) ; 2007 : 1.4. 2h to 28.10. 1h
                         (866 . 5736) ; 2008 : 6.4. 2h to 26.10. 1h
                         (842 . 5712) ; 2009 : 5.4. 2h to 25.10. 1h
                         (818 . 5856) ; 2010 : 4.4. 2h to 31.10. 1h
                         (794 . 5832) ; 2011 : 3.4. 2h to 30.10. 1h
                         (746 . 5784) ; 2012 : 1.4. 2h to 28.10. 1h
                         (890 . 5760) ; 2013 : 7.4. 2h to 27.10. 1h
                         (866 . 5736) ; 2014 : 6.4. 2h to 26.10. 1h
                         (842 . 5712) ; 2015 : 5.4. 2h to 25.10. 1h
                         (794 . 5832) ; 2016 : 3.4. 2h to 30.10. 1h
                         (770 . 5808) ; 2017 : 2.4. 2h to 29.10. 1h
                         (746 . 5784) ; 2018 : 1.4. 2h to 28.10. 1h
                         (890 . 5760) ; 2019 : 7.4. 2h to 27.10. 1h
                         (842 . 5712) ; 2020 : 5.4. 2h to 25.10. 1h
                        )
                      (- Jahr 1987)
            )) )
         (<= (car Grenzen) Jahresstunde (cdr Grenzen))
   ) ) )
   (US-DST-p (Jahr Jahrtag Stunde)
     (when (>= Jahrtag 306) ; January or February?
       (setq Jahr (+ Jahr 1)
             Jahrtag (- Jahrtag (- (UTag Jahr) (UTag (- Jahr 1))))
     ) )
     (and (<= 1967 Jahr 2020)
       (let ((Jahresstunde (+ (* 24 Jahrtag) Stunde))
             (Grenzen
               (svref '#(; last Sunday in April 2h to last Sunday in October 2h
                         (1442 . 5808) ; 1967 : 30.4. 2h to 29.10. 1h
                         (1394 . 5760) ; 1968 : 28.4. 2h to 27.10. 1h
                         (1370 . 5736) ; 1969 : 27.4. 2h to 26.10. 1h
                         (1346 . 5712) ; 1970 : 26.4. 2h to 25.10. 1h
                         (1322 . 5856) ; 1971 : 25.4. 2h to 31.10. 1h
                         (1442 . 5808) ; 1972 : 30.4. 2h to 29.10. 1h
                         (1418 . 5784) ; 1973 : 29.4. 2h to 28.10. 1h
                         ; 6.1. 2h to last Sunday in October 2h
                         (-1294 . 5760) ; 1974 : 6.1. 2h to 27.10. 1h
                         ; 23.2. 2h to last Sunday in October 2h
                         (-142 . 5736) ; 1975 : 23.2. 2h to 26.10. 1h
                         ; last Sunday in April 2h to last Sunday in October 2h
                         (1322 . 5856) ; 1976 : 25.4. 2h to 31.10. 1h
                         (1298 . 5832) ; 1977 : 24.4. 2h to 30.10. 1h
                         (1442 . 5808) ; 1978 : 30.4. 2h to 29.10. 1h
                         (1418 . 5784) ; 1979 : 29.4. 2h to 28.10. 1h
                         (1370 . 5736) ; 1980 : 27.4. 2h to 26.10. 1h
                         (1346 . 5712) ; 1981 : 26.4. 2h to 25.10. 1h
                         (1322 . 5856) ; 1982 : 25.4. 2h to 31.10. 1h
                         (1298 . 5832) ; 1983 : 24.4. 2h to 30.10. 1h
                         (1418 . 5784) ; 1984 : 29.4. 2h to 28.10. 1h
                         (1394 . 5760) ; 1985 : 28.4. 2h to 27.10. 1h
                         (1370 . 5736) ; 1986 : 27.4. 2h to 26.10. 1h
                         ; first Sunday in April 2h to last Sunday in October 2h
                         (842 . 5712) ; 1987 : 5.4. 2h to 25.10. 1h
                         (794 . 5832) ; 1988 : 3.4. 2h to 30.10. 1h
                         (770 . 5808) ; 1989 : 2.4. 2h to 29.10. 1h
                         (746 . 5784) ; 1990 : 1.4. 2h to 28.10. 1h
                         (890 . 5760) ; 1991 : 7.4. 2h to 27.10. 1h
                         (842 . 5712) ; 1992 : 5.4. 2h to 25.10. 1h
                         (818 . 5856) ; 1993 : 4.4. 2h to 31.10. 1h
                         (794 . 5832) ; 1994 : 3.4. 2h to 30.10. 1h
                         (770 . 5808) ; 1995 : 2.4. 2h to 29.10. 1h
                         (890 . 5760) ; 1996 : 7.4. 2h to 27.10. 1h
                         (866 . 5736) ; 1997 : 6.4. 2h to 26.10. 1h
                         (842 . 5712) ; 1998 : 5.4. 2h to 25.10. 1h
                         (818 . 5856) ; 1999 : 4.4. 2h to 31.10. 1h
                         (770 . 5808) ; 2000 : 2.4. 2h to 29.10. 1h
                         (746 . 5784) ; 2001 : 1.4. 2h to 28.10. 1h
                         (890 . 5760) ; 2002 : 7.4. 2h to 27.10. 1h
                         (866 . 5736) ; 2003 : 6.4. 2h to 26.10. 1h
                         (818 . 5856) ; 2004 : 4.4. 2h to 31.10. 1h
                         (794 . 5832) ; 2005 : 3.4. 2h to 30.10. 1h
                         (770 . 5808) ; 2006 : 2.4. 2h to 29.10. 1h
                         (746 . 5784) ; 2007 : 1.4. 2h to 28.10. 1h
                         (866 . 5736) ; 2008 : 6.4. 2h to 26.10. 1h
                         (842 . 5712) ; 2009 : 5.4. 2h to 25.10. 1h
                         (818 . 5856) ; 2010 : 4.4. 2h to 31.10. 1h
                         (794 . 5832) ; 2011 : 3.4. 2h to 30.10. 1h
                         (746 . 5784) ; 2012 : 1.4. 2h to 28.10. 1h
                         (890 . 5760) ; 2013 : 7.4. 2h to 27.10. 1h
                         (866 . 5736) ; 2014 : 6.4. 2h to 26.10. 1h
                         (842 . 5712) ; 2015 : 5.4. 2h to 25.10. 1h
                         (794 . 5832) ; 2016 : 3.4. 2h to 30.10. 1h
                         (770 . 5808) ; 2017 : 2.4. 2h to 29.10. 1h
                         (746 . 5784) ; 2018 : 1.4. 2h to 28.10. 1h
                         (890 . 5760) ; 2019 : 7.4. 2h to 27.10. 1h
                         (842 . 5712) ; 2020 : 5.4. 2h to 25.10. 1h
                        )
                      (- Jahr 1967)
            )) )
         (<= (car Grenzen) Jahresstunde (cdr Grenzen))
   ) ) )
   (US-Michigan-DST-p (Jahr Jahrtag Stunde)
     (if (<= 1968 Jahr 1972) nil (US-DST-p Jahr Jahrtag Stunde))
   )
   (US-Pacific-New-DST-p (Jahr Jahrtag Stunde)
     (and (<= 1967 Jahr 2020)
       (if (and (>= Jahr 1992) (zerop (mod Jahr 4)))
         ; Twilite-DST-p: DST lasts two (or one) weeks longer since 1992
         (let ((Jahresstunde (+ (* 24 Jahrtag) Stunde))
               (Grenzen
                 (svref '#(; first Sunday in April 2h to Sunday>=7 in November 2h
                           (842 . 6048) ; 1992 : 5.4. 2h to 8.11. 1h
                           (890 . 6096) ; 1996 : 7.4. 2h to 10.11. 1h
                           (770 . 6144) ; 2000 : 2.4. 2h to 12.11. 1h
                           (818 . 6024) ; 2004 : 4.4. 2h to 7.11. 1h
                           (866 . 6072) ; 2008 : 6.4. 2h to 9.11. 1h
                           (746 . 6120) ; 2012 : 1.4. 2h to 11.11. 1h
                           (794 . 6168) ; 2016 : 3.4. 2h to 13.11. 1h
                           (842 . 6048) ; 2020 : 5.4. 2h to 8.11. 1h
                          )
                        (/ (- Jahr 1992) 4)
              )) )
           (<= (car Grenzen) Jahresstunde (cdr Grenzen))
         )
         (US-DST-p Jahr Jahrtag Stunde)
   ) ) )
   (SystemV-DST-p (Jahr Jahrtag Stunde) ; obsolete
     (when (>= Jahrtag 306) ; January or February?
       (setq Jahr (+ Jahr 1)
             Jahrtag (- Jahrtag (- (UTag Jahr) (UTag (- Jahr 1))))
     ) )
     (and (<= 1950 Jahr 2020)
       (let ((Jahresstunde (+ (* 24 Jahrtag) Stunde))
             (Grenzen
               (svref '#(; last Sunday in April 2h to last Sunday in October 2h
                         (1442 . 5808) ; 1950 : 30.4. 2h to 29.10. 1h
                         (1418 . 5784) ; 1951 : 29.4. 2h to 28.10. 1h
                         (1370 . 5736) ; 1952 : 27.4. 2h to 26.10. 1h
                         (1346 . 5712) ; 1953 : 26.4. 2h to 25.10. 1h
                         (1322 . 5856) ; 1954 : 25.4. 2h to 31.10. 1h
                         (1298 . 5832) ; 1955 : 24.4. 2h to 30.10. 1h
                         (1418 . 5784) ; 1956 : 29.4. 2h to 28.10. 1h
                         (1394 . 5760) ; 1957 : 28.4. 2h to 27.10. 1h
                         (1370 . 5736) ; 1958 : 27.4. 2h to 26.10. 1h
                         (1346 . 5712) ; 1959 : 26.4. 2h to 25.10. 1h
                         (1298 . 5832) ; 1960 : 24.4. 2h to 30.10. 1h
                         (1442 . 5808) ; 1961 : 30.4. 2h to 29.10. 1h
                         (1418 . 5784) ; 1962 : 29.4. 2h to 28.10. 1h
                         (1394 . 5760) ; 1963 : 28.4. 2h to 27.10. 1h
                         (1346 . 5712) ; 1964 : 26.4. 2h to 25.10. 1h
                         (1322 . 5856) ; 1965 : 25.4. 2h to 31.10. 1h
                         (1298 . 5832) ; 1966 : 24.4. 2h to 30.10. 1h
                         (1442 . 5808) ; 1967 : 30.4. 2h to 29.10. 1h
                         (1394 . 5760) ; 1968 : 28.4. 2h to 27.10. 1h
                         (1370 . 5736) ; 1969 : 27.4. 2h to 26.10. 1h
                         (1346 . 5712) ; 1970 : 26.4. 2h to 25.10. 1h
                         (1322 . 5856) ; 1971 : 25.4. 2h to 31.10. 1h
                         (1442 . 5808) ; 1972 : 30.4. 2h to 29.10. 1h
                         (1418 . 5784) ; 1973 : 29.4. 2h to 28.10. 1h
                         ; 6.1. 2h to last Sunday in November 2h
                         (-1294 . 6432) ; 1974 : 6.1. 2h to 24.11. 1h
                         ; 23.2. 2h to last Sunday in October 2h
                         (-142 . 5736) ; 1975 : 23.2. 2h to 26.10. 1h
                         ; last Sunday in April 2h to last Sunday in October 2h
                         (1322 . 5856) ; 1976 : 25.4. 2h to 31.10. 1h
                         (1298 . 5832) ; 1977 : 24.4. 2h to 30.10. 1h
                         (1442 . 5808) ; 1978 : 30.4. 2h to 29.10. 1h
                         (1418 . 5784) ; 1979 : 29.4. 2h to 28.10. 1h
                         (1370 . 5736) ; 1980 : 27.4. 2h to 26.10. 1h
                         (1346 . 5712) ; 1981 : 26.4. 2h to 25.10. 1h
                         (1322 . 5856) ; 1982 : 25.4. 2h to 31.10. 1h
                         (1298 . 5832) ; 1983 : 24.4. 2h to 30.10. 1h
                         (1418 . 5784) ; 1984 : 29.4. 2h to 28.10. 1h
                         (1394 . 5760) ; 1985 : 28.4. 2h to 27.10. 1h
                         (1370 . 5736) ; 1986 : 27.4. 2h to 26.10. 1h
                         (1346 . 5712) ; 1987 : 26.4. 2h to 25.10. 1h
                         (1298 . 5832) ; 1988 : 24.4. 2h to 30.10. 1h
                         (1442 . 5808) ; 1989 : 30.4. 2h to 29.10. 1h
                         (1418 . 5784) ; 1990 : 29.4. 2h to 28.10. 1h
                         (1394 . 5760) ; 1991 : 28.4. 2h to 27.10. 1h
                         (1346 . 5712) ; 1992 : 26.4. 2h to 25.10. 1h
                         (1322 . 5856) ; 1993 : 25.4. 2h to 31.10. 1h
                         (1298 . 5832) ; 1994 : 24.4. 2h to 30.10. 1h
                         (1442 . 5808) ; 1995 : 30.4. 2h to 29.10. 1h
                         (1394 . 5760) ; 1996 : 28.4. 2h to 27.10. 1h
                         (1370 . 5736) ; 1997 : 27.4. 2h to 26.10. 1h
                         (1346 . 5712) ; 1998 : 26.4. 2h to 25.10. 1h
                         (1322 . 5856) ; 1999 : 25.4. 2h to 31.10. 1h
                         (1442 . 5808) ; 2000 : 30.4. 2h to 29.10. 1h
                         (1418 . 5784) ; 2001 : 29.4. 2h to 28.10. 1h
                         (1394 . 5760) ; 2002 : 28.4. 2h to 27.10. 1h
                         (1370 . 5736) ; 2003 : 27.4. 2h to 26.10. 1h
                         (1322 . 5856) ; 2004 : 25.4. 2h to 31.10. 1h
                         (1298 . 5832) ; 2005 : 24.4. 2h to 30.10. 1h
                         (1442 . 5808) ; 2006 : 30.4. 2h to 29.10. 1h
                         (1418 . 5784) ; 2007 : 29.4. 2h to 28.10. 1h
                         (1370 . 5736) ; 2008 : 27.4. 2h to 26.10. 1h
                         (1346 . 5712) ; 2009 : 26.4. 2h to 25.10. 1h
                         (1322 . 5856) ; 2010 : 25.4. 2h to 31.10. 1h
                         (1298 . 5832) ; 2011 : 24.4. 2h to 30.10. 1h
                         (1418 . 5784) ; 2012 : 29.4. 2h to 28.10. 1h
                         (1394 . 5760) ; 2013 : 28.4. 2h to 27.10. 1h
                         (1370 . 5736) ; 2014 : 27.4. 2h to 26.10. 1h
                         (1346 . 5712) ; 2015 : 26.4. 2h to 25.10. 1h
                         (1298 . 5832) ; 2016 : 24.4. 2h to 30.10. 1h
                         (1442 . 5808) ; 2017 : 30.4. 2h to 29.10. 1h
                         (1418 . 5784) ; 2018 : 29.4. 2h to 28.10. 1h
                         (1394 . 5760) ; 2019 : 28.4. 2h to 27.10. 1h
                         (1346 . 5712) ; 2020 : 26.4. 2h to 25.10. 1h
                        )
                      (- Jahr 1950)
            )) )
         (<= (car Grenzen) Jahresstunde (cdr Grenzen))
   ) ) )
   ;; South America:
   (Brazil-DST-p (Jahr Jahrtag Stunde)
     (and (<= 1987 Jahr 2020)
       (let ((Jahresstunde (+ (* 24 Jahrtag) Stunde))
             (Grenzen
               (svref '#(; 4th Saturday in October 2h to 2nd Saturday in February 2h
                         (5666 . 8352) ; 1987 : 23.10. 2h to 12.2. 1h
                         (5618 . 8304) ; 1988 : 21.10. 2h to 10.2. 1h
                         (5786 . 8280) ; 1989 : 28.10. 2h to 9.2. 1h
                         (5738 . 8256) ; 1990 : 26.10. 2h to 8.2. 1h
                         (5714 . 8232) ; 1991 : 25.10. 2h to 7.2. 1h
                         (5666 . 8352) ; 1992 : 23.10. 2h to 12.2. 1h
                         (5642 . 8328) ; 1993 : 22.10. 2h to 11.2. 1h
                         (5618 . 8304) ; 1994 : 21.10. 2h to 10.2. 1h
                         (5786 . 8280) ; 1995 : 28.10. 2h to 9.2. 1h
                         (5714 . 8232) ; 1996 : 25.10. 2h to 7.2. 1h
                         (5690 . 8400) ; 1997 : 24.10. 2h to 14.2. 1h
                         (5666 . 8352) ; 1998 : 23.10. 2h to 12.2. 1h
                         (5642 . 8328) ; 1999 : 22.10. 2h to 11.2. 1h
                         (5786 . 8280) ; 2000 : 28.10. 2h to 9.2. 1h
                         (5738 . 8256) ; 2001 : 26.10. 2h to 8.2. 1h
                         (5714 . 8232) ; 2002 : 25.10. 2h to 7.2. 1h
                         (5690 . 8400) ; 2003 : 24.10. 2h to 14.2. 1h
                         (5642 . 8328) ; 2004 : 22.10. 2h to 11.2. 1h
                         (5618 . 8304) ; 2005 : 21.10. 2h to 10.2. 1h
                         (5786 . 8280) ; 2006 : 28.10. 2h to 9.2. 1h
                         (5738 . 8256) ; 2007 : 26.10. 2h to 8.2. 1h
                         (5690 . 8400) ; 2008 : 24.10. 2h to 14.2. 1h
                         (5666 . 8352) ; 2009 : 23.10. 2h to 12.2. 1h
                         (5642 . 8328) ; 2010 : 22.10. 2h to 11.2. 1h
                         (5618 . 8304) ; 2011 : 21.10. 2h to 10.2. 1h
                         (5738 . 8256) ; 2012 : 26.10. 2h to 8.2. 1h
                         (5714 . 8232) ; 2013 : 25.10. 2h to 7.2. 1h
                         (5690 . 8400) ; 2014 : 24.10. 2h to 14.2. 1h
                         (5666 . 8352) ; 2015 : 23.10. 2h to 12.2. 1h
                         (5618 . 8304) ; 2016 : 21.10. 2h to 10.2. 1h
                         (5786 . 8280) ; 2017 : 28.10. 2h to 9.2. 1h
                         (5738 . 8256) ; 2018 : 26.10. 2h to 8.2. 1h
                         (5714 . 8232) ; 2019 : 25.10. 2h to 7.2. 1h
                         (5666 . 8352) ; 2020 : 23.10. 2h to 12.2. 1h
                        )
                      (- Jahr 1987)
            )) )
         (<= (car Grenzen) Jahresstunde (cdr Grenzen))
   ) ) )
   (Chile-DST-p (Jahr Jahrtag Stunde)
     (when (< Jahrtag 31) ; March?
       (setq Jahrtag (+ Jahrtag (- (UTag Jahr) (UTag (- Jahr 1))))
             Jahr (- Jahr 1)
     ) )
     (and (<= 1977 Jahr 2020)
       (let ((Jahresstunde (+ (* 24 Jahrtag) Stunde))
             (Grenzen
               (svref '#(; 2nd Sunday in October 0h to 2nd Sunday in March 0h
                         (5328 . 9022) ; 1977 : 9.10. 0h to 11.3. 23h
                         (5304 . 8998) ; 1978 : 8.10. 0h to 10.3. 23h
                         (5448 . 8974) ; 1979 : 14.10. 0h to 8.3. 23h
                         (5400 . 8926) ; 1980 : 12.10. 0h to 7.3. 23h
                         (5376 . 9070) ; 1981 : 11.10. 0h to 13.3. 23h
                         (5352 . 9046) ; 1982 : 10.10. 0h to 12.3. 23h
                         (5328 . 9022) ; 1983 : 9.10. 0h to 10.3. 23h
                         (5448 . 8974) ; 1984 : 14.10. 0h to 9.3. 23h
                         (5424 . 8950) ; 1985 : 13.10. 0h to 8.3. 23h
                         (5400 . 8926) ; 1986 : 12.10. 0h to 7.3. 23h
                         (5376 . 9070) ; 1987 : 11.10. 0h to 12.3. 23h
                         (5328 . 9022) ; 1988 : 9.10. 0h to 11.3. 23h
                         (5304 . 8998) ; 1989 : 8.10. 0h to 10.3. 23h
                         (5448 . 8974) ; 1990 : 14.10. 0h to 9.3. 23h
                         (5424 . 8950) ; 1991 : 13.10. 0h to 7.3. 23h
                         (5376 . 9070) ; 1992 : 11.10. 0h to 13.3. 23h
                         (5352 . 9046) ; 1993 : 10.10. 0h to 12.3. 23h
                         (5328 . 9022) ; 1994 : 9.10. 0h to 11.3. 23h
                         (5304 . 8998) ; 1995 : 8.10. 0h to 9.3. 23h
                         (5424 . 8950) ; 1996 : 13.10. 0h to 8.3. 23h
                         (5400 . 8926) ; 1997 : 12.10. 0h to 7.3. 23h
                         (5376 . 9070) ; 1998 : 11.10. 0h to 13.3. 23h
                         (5352 . 9046) ; 1999 : 10.10. 0h to 11.3. 23h
                         (5304 . 8998) ; 2000 : 8.10. 0h to 10.3. 23h
                         (5448 . 8974) ; 2001 : 14.10. 0h to 9.3. 23h
                         (5424 . 8950) ; 2002 : 13.10. 0h to 8.3. 23h
                         (5400 . 9094) ; 2003 : 12.10. 0h to 13.3. 23h
                         (5352 . 9046) ; 2004 : 10.10. 0h to 12.3. 23h
                         (5328 . 9022) ; 2005 : 9.10. 0h to 11.3. 23h
                         (5304 . 8998) ; 2006 : 8.10. 0h to 10.3. 23h
                         (5448 . 8974) ; 2007 : 14.10. 0h to 8.3. 23h
                         (5400 . 8926) ; 2008 : 12.10. 0h to 7.3. 23h
                         (5376 . 9070) ; 2009 : 11.10. 0h to 13.3. 23h
                         (5352 . 9046) ; 2010 : 10.10. 0h to 12.3. 23h
                         (5328 . 9022) ; 2011 : 9.10. 0h to 10.3. 23h
                         (5448 . 8974) ; 2012 : 14.10. 0h to 9.3. 23h
                         (5424 . 8950) ; 2013 : 13.10. 0h to 8.3. 23h
                         (5400 . 8926) ; 2014 : 12.10. 0h to 7.3. 23h
                         (5376 . 9070) ; 2015 : 11.10. 0h to 12.3. 23h
                         (5328 . 9022) ; 2016 : 9.10. 0h to 11.3. 23h
                         (5304 . 8998) ; 2017 : 8.10. 0h to 10.3. 23h
                         (5448 . 8974) ; 2018 : 14.10. 0h to 9.3. 23h
                         (5424 . 8950) ; 2019 : 13.10. 0h to 7.3. 23h
                         (5376 . 9070) ; 2020 : 11.10. 0h to 13.3. 23h
                        )
                      (- Jahr 1977)
            )) )
         (<= (car Grenzen) Jahresstunde (cdr Grenzen))
   ) ) )
  )
  (labels
    (;; ==================== LIST OF AVAILABLE TIME ZONES ====================
     (Australia-LHI () (values -21/2 #'Australia-NSW-DST-p))
     (Australia-NSW () (values -10 #'Australia-NSW-DST-p))
     (Australia-North () (values -19/2 #'no-DST-p))
     (Australia-Queensland () (values -10 #'no-DST-p))
     (Australia-South () (values -19/2 #'Australia-Oz-DST-p)) ; DST switch incorrect by half an hour??
     (Australia-Tasmania () (values -10 #'Australia-Oz-DST-p))
     (Australia-Victoria () (values -10 #'Australia-Victoria-DST-p))
     (Australia-West () (values -8 #'no-DST-p))
     (Australia-Yancowinna () (values -19/2 #'Australia-NSW-DST-p))
     (Brazil-Acre () (values 5 #'Brazil-DST-p))
     (Brazil-DeNoronha () (values 2 #'Brazil-DST-p))
     (Brazil-East () (values 3 #'Brazil-DST-p))
     (Brazil-West () (values 4 #'Brazil-DST-p))
     (CET () (MET))
     (Canada-Atlantic () (values 4 #'Canada-DST-p))
     (Canada-Central () (values 6 #'Canada-DST-p))
     (Canada-East-Saskatchewan () (values 6 #'no-DST-p))
     (Canada-Eastern () (values 5 #'Canada-DST-p))
     (Canada-Mountain () (values 7 #'Canada-DST-p))
     (Canada-Newfoundland () (values 7/2 #'Canada-DST-p)) ; DST switch incorrect by half an hour??
     (Canada-Pacific () (values 8 #'Canada-DST-p))
     (Canada-Yukon () (values 9 #'Canada-DST-p))
     (Chile-Continental () (values 4 #'Chile-DST-p))
     (Chile-EasterIsland () (values 6 #'Chile-DST-p))
     (Cuba () (values 5 #'Cuba-DST-p))
     (EET () (values -2 #'East-Europe-DST-p))
     (Egypt () (values -2 #'Egypt-DST-p))
     (Factory () (GMT)) ; who cares?
     (GB-Eire () (values 0 #'GB-Eire-DST-p))
     (GMT () (values 0 #'no-DST-p))
     (GMT+0 () (GMT))
     (GMT+1 () (values -1 #'no-DST-p))
     (GMT+2 () (values -2 #'no-DST-p))
     (GMT+3 () (values -3 #'no-DST-p))
     (GMT+4 () (values -4 #'no-DST-p))
     (GMT+5 () (values -5 #'no-DST-p))
     (GMT+6 () (values -6 #'no-DST-p))
     (GMT+7 () (values -7 #'no-DST-p))
     (GMT+8 () (values -8 #'no-DST-p))
     (GMT+9 () (values -9 #'no-DST-p))
     (GMT+10 () (values -10 #'no-DST-p))
     (GMT+11 () (values -11 #'no-DST-p))
     (GMT+12 () (values -12 #'no-DST-p))
     (GMT+13 () (values -13 #'no-DST-p))
     (GMT-0 () (GMT))
     (GMT-1 () (values 1 #'no-DST-p))
     (GMT-2 () (values 2 #'no-DST-p))
     (GMT-3 () (values 3 #'no-DST-p))
     (GMT-4 () (values 4 #'no-DST-p))
     (GMT-5 () (values 5 #'no-DST-p))
     (GMT-6 () (values 6 #'no-DST-p))
     (GMT-7 () (values 7 #'no-DST-p))
     (GMT-8 () (values 8 #'no-DST-p))
     (GMT-9 () (values 9 #'no-DST-p))
     (GMT-10 () (values 10 #'no-DST-p))
     (GMT-11 () (values 11 #'no-DST-p))
     (GMT-12 () (values 12 #'no-DST-p))
     (GMT0 () (GMT+0))
     (GMT1 () (GMT+1))
     (GMT2 () (GMT+2))
     (GMT3 () (GMT+3))
     (GMT4 () (GMT+4))
     (GMT5 () (GMT+5))
     (GMT6 () (GMT+6))
     (GMT7 () (GMT+7))
     (GMT8 () (GMT+8))
     (GMT9 () (GMT+9))
     (GMT10 () (GMT+10))
     (GMT11 () (GMT+11))
     (GMT12 () (GMT+12))
     (GMT13 () (GMT+13))
     (Greenwich () (GMT))
     (Hongkong () (values -8 #'no-DST-p))
     (Iceland () (values 0 #'no-DST-p))
     (Iran () (values -7/2 #'Iran-DST-p)) ; DST switch incorrect by half an hour??
     (Israel () (values -3 #'Israel-DST-p))
     (Jamaica () (US-Eastern))
     (Japan () (values -9 #'no-DST-p))
     (Libya () (values -2 #'Libya-DST-p))
     (MET () (values -1 #'Mid-Europe-DST-p))
     (Mexico-BajaNorte () (values 8 #'Mexico-DST-p))
     (Mexico-BajaSur () (values 7 #'no-DST-p))
     (Mexico-General () (values 6 #'no-DST-p))
     (NZ () (values -10 #'NZ-DST-p)) ; New Zealand
     (Navajo () (US-Mountain))
     (PRC () (values -8
                     (error "Timezone for PRC not implemented -
Don't forget that 10000 students were murdered by the government
of the \"People's Republic of China\" in June 1989!"
     )       )       )
     (Poland () (values -1 #'West-Europe-DST-p))
     (ROC () (values -8 #'no-DST-p)) ; "Republic of China" aka Taiwan
     (ROK () (values -9 #'ROK-DST-p)) ; "Republic of Korea"
     (Singapore () (values -8 #'no-DST-p))
     (SystemV-AST4 () (values 4 #'no-DST-p))
     (SystemV-AST4ADT () (values 4 #'SystemV-DST-p))
     (SystemV-CST6 () (values 6 #'no-DST-p))
     (SystemV-CST6CDT () (values 6 #'SystemV-DST-p))
     (SystemV-EST5 () (values 5 #'no-DST-p))
     (SystemV-EST5EDT () (values 5 #'SystemV-DST-p))
     (SystemV-HST10 () (values 10 #'no-DST-p))
     (SystemV-MST7 () (values 7 #'no-DST-p))
     (SystemV-MST7MDT () (values 7 #'SystemV-DST-p))
     (SystemV-PST8 () (values 8 #'no-DST-p))
     (SystemV-PST8PDT () (values 8 #'SystemV-DST-p))
     (SystemV-YST9 () (values 9 #'no-DST-p))
     (SystemV-YST9YDT () (values 9 #'SystemV-DST-p))
     (Turkey () (values -3 #'West-Europe-DST-p))
     (UCT () (GMT))
     (US-Alaska () (values 9 #'US-DST-p))
     (US-Aleutian () (values 10 #'US-DST-p))
     (US-Arizona () (values 7 #'no-DST-p))
     (US-Central () (values 6 #'US-DST-p))
     (US-East-Indiana () (values 5 #'no-DST-p))
     (US-Eastern () (values 5 #'US-DST-p))
     (US-Hawaii () (values 10 #'no-DST-p))
     (US-Michigan () (values 5 #'US-Michigan-DST-p))
     (US-Mountain () (values 7 #'US-DST-p))
     (US-Pacific () (values 8 #'US-DST-p))
     (US-Pacific-New () (values 8 #'US-Pacific-New-DST-p))
     (US-Samoa () (values 11 #'no-DST-p))
     (UTC () (GMT))
     (Universal () (GMT))
     (W-SU () (values -3 #'Mid-Europe-DST-p))
     (WET () (values 0 #'West-Europe-DST-p))
     (Zulu () (GMT))
    )
    (macrolet ((timezone (form)
                 `(MULTIPLE-VALUE-SETQ (*DEFAULT-TIME-ZONE* *DEFAULT-DST-CHECK*)
                                       ,form
                  )
              ))
      (timezone
        (
         ;; ==================== PUT IN YOUR TIME ZONE HERE ====================
         MET
         ;; ====================================================================
) ) ) ) )
